package net.rocketpowered.connector.server.commands;

import org.bson.types.ObjectId;
import com.mojang.brigadier.CommandDispatcher;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import net.minecraft.commands.CommandSourceStack;
import net.minecraft.commands.Commands;
import net.minecraft.network.chat.TranslatableComponent;
import net.minecraft.world.entity.player.Player;
import net.rocketpowered.common.Punishment;
import net.rocketpowered.connector.capability.ServerRocketHandle;
import net.rocketpowered.connector.commands.RocketCommands;
import net.rocketpowered.connector.commands.arguments.ObjectIdArgument;
import reactor.core.publisher.Mono;
import reactor.core.scheduler.Schedulers;

public class PardonCommand {

  public static void register(CommandDispatcher<CommandSourceStack> dispatcher) {
    dispatcher.register(Commands.literal("pardon")
        .requires(source -> source.getEntity() instanceof Player
            && ServerRocketHandle.get(source.getEntity()).isPresent())
        .then(Commands.argument("punishment_id", ObjectIdArgument.objectId())
            .executes(context -> pardon(context.getSource(),
                context.getArgument("punishment_id", ObjectId.class)))));
  }

  private static int pardon(CommandSourceStack source, ObjectId punishmentId)
      throws CommandSyntaxException {
    var handle = ServerRocketHandle.get(source.getEntity())
        .orElseThrow(RocketCommands.ROCKET_UNAVAILABLE_ERROR::create);
    handle.getGateway().pardon(handle.getUserId(), punishmentId)
        .switchIfEmpty(Mono.<Punishment>empty()
            .doOnSuccess(__ -> source.getServer().execute(() -> source.sendFailure(
                new TranslatableComponent("commands.rocket.pardon.unknown_punishment")))))
        .doOnError(error -> source.getServer().execute(() -> source.sendFailure(
            new TranslatableComponent("commands.rocket.pardon.failure", error.getMessage()))))
        .onErrorResume(__ -> Mono.empty())
        .doOnNext(punishment -> source.getServer().execute(() -> source.sendSuccess(
            new TranslatableComponent("commands.rocket.pardon.success",
                punishment.punished().displayName()),
            true)))
        .subscribeOn(Schedulers.boundedElastic())
        .subscribe();
    return 1;
  }
}
