package net.rocketpowered.connector.server;

import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Instant;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.time.format.FormatStyle;
import java.util.Locale;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import org.bson.types.ObjectId;
import org.jetbrains.annotations.Nullable;
import org.slf4j.Logger;
import com.github.benmanes.caffeine.cache.RemovalCause;
import com.google.common.base.Strings;
import com.mojang.logging.LogUtils;
import io.netty.buffer.Unpooled;
import net.minecraft.ChatFormatting;
import net.minecraft.Util;
import net.minecraft.core.Direction;
import net.minecraft.network.FriendlyByteBuf;
import net.minecraft.network.chat.Component;
import net.minecraft.network.chat.TranslatableComponent;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.server.network.ServerLoginPacketListenerImpl;
import net.minecraft.world.entity.Entity;
import net.minecraftforge.common.MinecraftForge;
import net.minecraftforge.common.capabilities.Capability;
import net.minecraftforge.common.capabilities.ICapabilityProvider;
import net.minecraftforge.common.util.LazyOptional;
import net.minecraftforge.event.AttachCapabilitiesEvent;
import net.minecraftforge.event.ServerChatEvent;
import net.minecraftforge.event.TickEvent;
import net.minecraftforge.event.TickEvent.PlayerTickEvent;
import net.minecraftforge.event.entity.player.PlayerEvent;
import net.minecraftforge.event.entity.player.PlayerNegotiationEvent;
import net.minecraftforge.event.server.ServerAboutToStartEvent;
import net.minecraftforge.eventbus.api.SubscribeEvent;
import net.minecraftforge.network.PacketDistributor;
import net.minecraftforge.server.ServerLifecycleHooks;
import net.rocketpowered.common.network.Connection;
import net.rocketpowered.common.network.protocol.v1.gameserver.GameServerProtocol;
import net.rocketpowered.common.network.protocol.v1.login.GameServerLoginMessage;
import net.rocketpowered.connector.ModDist;
import net.rocketpowered.connector.RocketConnector;
import net.rocketpowered.connector.capability.RocketHandle;
import net.rocketpowered.connector.capability.ServerRocketHandle;
import net.rocketpowered.connector.network.InvalidateHandleMessage;
import net.rocketpowered.connector.network.NetworkManager;
import net.rocketpowered.connector.network.RegisterHandleMessage;
import net.rocketpowered.connector.network.SyncHandleMessage;
import net.rocketpowered.sdk.Rocket;
import net.rocketpowered.sdk.network.protocol.gameserver.GameServerConnectionHandler;
import reactor.core.publisher.Mono;

public class ServerDist implements ModDist {

  private static final Logger logger = LogUtils.getLogger();

  private static final Path PROPERTIES_PATH = Paths.get("rocket.properties");

  private static final DateTimeFormatter formatter =
      DateTimeFormatter.ofLocalizedDateTime(FormatStyle.SHORT)
          .withLocale(Locale.getDefault())
          .withZone(ZoneId.systemDefault());

  private RocketProperties properties;

  private final Map<UUID, LazyOptional<ServerRocketHandle>> playerHandles =
      new ConcurrentHashMap<>();

  public ServerDist() {
    MinecraftForge.EVENT_BUS.register(this);

    // TODO Class load this to fix: https://github.com/MinecraftForge/EventBus/issues/44
    System.out.println("Class loaded: " + GameServerProtocol.class);
    System.out.println("Class loaded: " + RemovalCause.class);
  }

  @Override
  public boolean setup() {
    this.properties = RocketProperties.fromFile(PROPERTIES_PATH);
    this.properties.store(PROPERTIES_PATH);

    if (Strings.isNullOrEmpty(this.properties.serverId)) {
      logger.warn("No server ID specified.");
      return false;
    }

    if (!ObjectId.isValid(this.properties.serverId)) {
      logger.warn("Invalid server ID: {}", this.properties.serverId);
      return false;
    }

    if (Strings.isNullOrEmpty(this.properties.serverToken)) {
      logger.warn("No server token specified.");
      return false;
    }

    if (!ServerLifecycleHooks.getCurrentServer().usesAuthentication()) {
      logger.warn("Server running in offline mode, Rocket requires online mode to be enabled.");
      return false;
    }

    return true;
  }

  @Override
  public Mono<Void> login(Connection connection) {
    return Mono
        .when(connection.send(new GameServerLoginMessage(new ObjectId(this.properties.serverId),
            this.properties.serverToken)))
        .doOnSuccess(
            __ -> connection.loadConnectionHandler(new GameServerConnectionHandler(connection)));
  }

  @SubscribeEvent
  public void handleServerAboutToStart(ServerAboutToStartEvent event) {
    RocketConnector.getInstance().init();
  }

  @SubscribeEvent
  public void handleAttachCapabilities(AttachCapabilitiesEvent<Entity> event) {
    if (event.getObject() instanceof ServerPlayer player) {
      // Cannot access player ID during this event as the player has not been fully constructed yet.
      event.addCapability(new ResourceLocation(RocketConnector.ID, "handle"),
          new ICapabilityProvider() {

            @Override
            public <T> LazyOptional<T> getCapability(Capability<T> cap, Direction side) {
              return cap == RocketHandle.CAPABILITY
                  ? ServerDist.this.playerHandles
                      .getOrDefault(player.getUUID(), LazyOptional.empty())
                      .cast()
                  : LazyOptional.empty();
            }
          });
    }
  }

  @SubscribeEvent
  public void handlePlayerNegotiation(PlayerNegotiationEvent event) {
    var gameProfile = event.getProfile();
    if (!RocketConnector.isActive() || !gameProfile.isComplete()) {
      return;
    }

    var listener = (ServerLoginPacketListenerImpl) event.getConnection().getPacketListener();

    Rocket.gameServerInterface().ifPresentOrElse(
        gateway -> event.enqueueWork(gateway.getUserId(gameProfile.getId())
            .flatMapMany(gateway::getActivePunishments)
            .filter(punishment -> punishment.type().isBan())
            .next()
            .doOnSubscribe(__ -> logger.info("Checking {}'s punishment records...",
                gameProfile.getName()))
            .doOnError(error -> {
              logger.error("Login error occurred, disconnecting: {}", gameProfile.getName());
              logger.error("Error: ", error);
              listener.disconnect(
                  new TranslatableComponent("multiplayer.disconnect.authservers_down"));
            })
            .doOnSuccess(ban -> {
              if (ban == null) {
                logger.info("{} has no active bans", gameProfile.getName());
              } else {
                logger.info("{} is banned (Punishment ID: {})",
                    gameProfile.getName(), ban.id());
                listener.disconnect(createBanMessage(ban.reason(), ban.expiresAt()));
              }
            })
            .then()
            .toFuture()),
        () -> {
          logger.error("Not connected to Rocket, disconnecting: {}", gameProfile.getName());
          listener.disconnect(
              new TranslatableComponent("multiplayer.disconnect.authservers_down"));
        });
  }

  @SubscribeEvent
  public void handlePlayerLoggedIn(PlayerEvent.PlayerLoggedInEvent event) {
    UUID playerId = event.getPlayer().getGameProfile().getId();
    if (playerId != null) {
      Rocket.gameServerInterfaceFeed()
          .flatMap(gateway -> gateway.getUserId(event.getPlayer().getUUID())
              .map(userId -> new ServerRocketHandle(
                  userId, (ServerPlayer) event.getPlayer(), gateway))
              .doOnNext(this::registerHandle)
              .delayUntil(
                  handle -> gateway.onClose().doOnSuccess(__ -> this.invalidateHandle(handle))))
          .subscribe();
    }
  }

  @SubscribeEvent
  public void handleStartTracking(PlayerEvent.StartTracking event) {
    ServerRocketHandle.get(event.getTarget())
        .ifPresent(handle -> {
          PacketDistributor.PacketTarget target =
              PacketDistributor.PLAYER.with(() -> (ServerPlayer) event.getPlayer());
          NetworkManager.channel.send(
              target, new RegisterHandleMessage(handle.getPlayer().getUUID(), handle.getUserId()));
          var out = new FriendlyByteBuf(Unpooled.buffer());
          handle.writeSyncData(out);
          NetworkManager.channel.send(
              target, new SyncHandleMessage(handle.getPlayer().getId(), out));
        });
  }

  @SubscribeEvent
  public void handlePlayerClone(PlayerEvent.Clone event) {
    ServerRocketHandle.get(event.getPlayer())
        .ifPresent(player -> player.setPlayer((ServerPlayer) event.getPlayer()));
  }

  private void registerHandle(ServerRocketHandle handle) {
    LazyOptional<ServerRocketHandle> oldHandle =
        this.playerHandles.put(handle.getPlayer().getUUID(), LazyOptional.of(() -> handle));
    handle.getPlayer().getServer().getCommands().sendCommands(handle.getPlayer());
    handle.getPlayer().refreshDisplayName();
    if (oldHandle != null) {
      oldHandle.ifPresent(ServerRocketHandle::dispose);
    }
    NetworkManager.channel.send(
        PacketDistributor.TRACKING_ENTITY_AND_SELF.with(handle::getPlayer),
        new RegisterHandleMessage(handle.getPlayer().getUUID(), handle.getUserId()));
  }

  private void invalidateHandle(ServerRocketHandle handle) {
    handle.dispose();
    this.playerHandles.remove(handle.getPlayer().getUUID());
    NetworkManager.channel.send(
        PacketDistributor.TRACKING_ENTITY_AND_SELF.with(handle::getPlayer),
        new InvalidateHandleMessage(handle.getPlayer().getUUID()));
  }

  @SubscribeEvent
  public void handlePlayerLoggedOut(PlayerEvent.PlayerLoggedOutEvent event) {
    ServerRocketHandle.get(event.getPlayer()).ifPresent(this::invalidateHandle);
  }

  @SubscribeEvent
  public void handlePlayerTick(PlayerTickEvent event) {
    if (event.phase == TickEvent.Phase.END) {
      ServerRocketHandle.get(event.player)
          .filter(RocketHandle::requiresSync)
          .ifPresent(handle -> {
            var out = new FriendlyByteBuf(Unpooled.buffer());
            handle.writeSyncData(out);
            NetworkManager.channel.send(
                PacketDistributor.TRACKING_ENTITY_AND_SELF.with(handle::getPlayer),
                new SyncHandleMessage(handle.getPlayer().getId(), out));
          });
    }
  }

  @SubscribeEvent
  public void handleServerChat(ServerChatEvent event) {
    var handle = event.getPlayer().getCapability(RocketHandle.CAPABILITY).orElse(null);
    if (handle != null && handle.isMuted()) {
      event.setCanceled(true);
      if (this.properties.displayMutedMessage) {
        event.getPlayer().sendMessage(
            new TranslatableComponent("chat.muted").withStyle(ChatFormatting.RED), Util.NIL_UUID);
      }
    }
  }

  @SubscribeEvent
  public void handleTabListNameFormat(PlayerEvent.TabListNameFormat event) {
    event.getPlayer().getCapability(RocketHandle.CAPABILITY)
        .map(handle -> handle.formatTabListName(event.getDisplayName()))
        .ifPresent(event::setDisplayName);
  }

  public static Component createBanMessage(String reason, @Nullable Instant expiresAt) {
    return new TranslatableComponent("rocket.message.disconnect.ban",
        reason, expiresAt == null
            ? new TranslatableComponent("ban.infinite")
            : formatter.format(expiresAt));
  }
}
