package net.rocketpowered.connector.client;

import java.math.BigInteger;
import java.security.MessageDigest;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import org.apache.commons.codec.digest.MessageDigestAlgorithms;
import org.bson.types.ObjectId;
import org.slf4j.Logger;
import com.mojang.logging.LogUtils;
import net.minecraft.ChatFormatting;
import net.minecraft.client.Minecraft;
import net.minecraft.client.player.AbstractClientPlayer;
import net.minecraft.core.Direction;
import net.minecraft.resources.ResourceLocation;
import net.minecraft.world.entity.Entity;
import net.minecraftforge.client.event.RenderNameplateEvent;
import net.minecraftforge.common.ForgeConfigSpec;
import net.minecraftforge.common.MinecraftForge;
import net.minecraftforge.common.capabilities.Capability;
import net.minecraftforge.common.capabilities.ICapabilityProvider;
import net.minecraftforge.common.util.LazyOptional;
import net.minecraftforge.event.AttachCapabilitiesEvent;
import net.minecraftforge.eventbus.api.SubscribeEvent;
import net.minecraftforge.fml.ModLoadingContext;
import net.minecraftforge.fml.config.ModConfig;
import net.minecraftforge.fml.event.lifecycle.FMLClientSetupEvent;
import net.minecraftforge.fml.javafmlmod.FMLJavaModLoadingContext;
import net.rocketpowered.common.network.Connection;
import net.rocketpowered.common.network.protocol.v1.login.GameClientHelloMessage;
import net.rocketpowered.common.network.protocol.v1.login.GameClientLoginMessage;
import net.rocketpowered.common.network.protocol.v1.login.GameClientReadyMessage;
import net.rocketpowered.common.network.protocol.v1.login.GameClientSecretMessage;
import net.rocketpowered.connector.ModDist;
import net.rocketpowered.connector.RocketConnector;
import net.rocketpowered.connector.capability.CommonRocketHandle;
import net.rocketpowered.connector.capability.RocketHandle;
import net.rocketpowered.sdk.network.protocol.gameclient.GameClientConnectionHandler;
import reactor.core.publisher.Mono;
import reactor.core.scheduler.Schedulers;

public class ClientDist implements ModDist {

  private static final ForgeConfigSpec configSpec;
  public static final ClientConfig config;

  static {
    var specPair = new ForgeConfigSpec.Builder().configure(ClientConfig::new);
    configSpec = specPair.getRight();
    config = specPair.getLeft();
  }

  private static final Logger logger = LogUtils.getLogger();

  private final Minecraft minecraft;

  private final Map<UUID, LazyOptional<RocketHandle>> playerHandles = new ConcurrentHashMap<>();

  public ClientDist() {
    this.minecraft = Minecraft.getInstance();

    var modEventBus = FMLJavaModLoadingContext.get().getModEventBus();
    modEventBus.addListener(this::handleClientSetup);

    ModLoadingContext.get().registerConfig(ModConfig.Type.CLIENT, configSpec);
    MinecraftForge.EVENT_BUS.register(this);
  }

  private void handleClientSetup(FMLClientSetupEvent event) {
    RocketConnector.getInstance().init();
  }

  @Override
  public Mono<Void> login(Connection connection) {
    return Mono.from(connection.send(new GameClientHelloMessage()))
        .cast(GameClientSecretMessage.class)
        .publishOn(Schedulers.boundedElastic())
        .handle((message, sink) -> {
          if (ClientDist.config.bypassAuthentication.get()) {
            sink.complete();
            return;
          }
          try {
            var messageDigest = MessageDigest.getInstance(MessageDigestAlgorithms.SHA_1);
            messageDigest.update(message.encodedSecretKey());
            var serverId = new BigInteger(messageDigest.digest()).toString(16);
            this.minecraft.getMinecraftSessionService().joinServer(
                this.minecraft.getUser().getGameProfile(),
                this.minecraft.getUser().getAccessToken(),
                serverId);
            sink.complete();
          } catch (Exception e) {
            sink.error(e);
          }
        })
        .then(Mono.from(
            connection.send(new GameClientLoginMessage(new ObjectId(config.projectId.get()),
                this.minecraft.getUser().getName()))))
        .cast(GameClientReadyMessage.class)
        .map(GameClientReadyMessage::user)
        .doOnNext(
            user -> logger.info("Successfully logged in to Rocket as: {}", user.displayName()))
        .map(user -> new GameClientConnectionHandler(connection, user))
        .doOnNext(connection::loadConnectionHandler)
        .then();
  }

  public void registerHandle(UUID playerId, ObjectId userId) {
    this.playerHandles.put(playerId, LazyOptional.of(() -> new CommonRocketHandle(userId)));
  }

  public void invalidateHandle(UUID playerId) {
    this.playerHandles.remove(playerId);
  }

  @SubscribeEvent
  public void handleRenderNameplate(RenderNameplateEvent event) {
    var handle = event.getEntity().getCapability(RocketHandle.CAPABILITY).orElse(null);
    var clientHandle = this.minecraft.player == null
        ? null
        : this.minecraft.player.getCapability(RocketHandle.CAPABILITY).orElse(null);
    if (handle != null && clientHandle != null) {
      var entityGuild = handle.getGuild();
      if (entityGuild.isPresent() && clientHandle.getGuild().equals(entityGuild)) {
        event.setContent(event.getContent().copy().withStyle(ChatFormatting.GREEN));
      }
    }
  }

  @SubscribeEvent
  public void handleAttachCapabilities(AttachCapabilitiesEvent<Entity> event) {
    if (event.getObject() instanceof AbstractClientPlayer player) {
      event.addCapability(new ResourceLocation(RocketConnector.ID, "handle"),
          new ICapabilityProvider() {

            @Override
            public <T> LazyOptional<T> getCapability(Capability<T> cap, Direction side) {
              return cap == RocketHandle.CAPABILITY
                  ? ClientDist.this.playerHandles
                      .getOrDefault(player.getUUID(), LazyOptional.empty())
                      .cast()
                  : LazyOptional.empty();
            }
          });
    }
  }
}
