package net.rocketpowered.connector.server.commands;

import java.time.Instant;
import java.util.Arrays;
import java.util.Collection;
import org.jetbrains.annotations.Nullable;
import com.mojang.authlib.GameProfile;
import com.mojang.brigadier.CommandDispatcher;
import com.mojang.brigadier.builder.ArgumentBuilder;
import com.mojang.brigadier.builder.LiteralArgumentBuilder;
import com.mojang.brigadier.exceptions.CommandSyntaxException;
import net.minecraft.ChatFormatting;
import net.minecraft.commands.CommandSourceStack;
import net.minecraft.commands.Commands;
import net.minecraft.commands.arguments.GameProfileArgument;
import net.minecraft.commands.arguments.MessageArgument;
import net.minecraft.network.chat.ClickEvent;
import net.minecraft.network.chat.Component;
import net.minecraft.network.chat.ComponentUtils;
import net.minecraft.network.chat.HoverEvent;
import net.minecraft.network.chat.TextComponent;
import net.minecraft.network.chat.TranslatableComponent;
import net.rocketpowered.common.PunishmentType;
import net.rocketpowered.connector.capability.ServerRocketHandle;
import net.rocketpowered.connector.commands.RocketCommands;
import net.rocketpowered.connector.commands.arguments.InstantArgument;
import reactor.core.publisher.Flux;
import reactor.core.publisher.Mono;
import reactor.core.scheduler.Schedulers;

public class PunishCommands {

  // To avoid conflicts with other ban/mute commands.
  private static final String PREFIX = "r";

  public static void register(CommandDispatcher<CommandSourceStack> dispatcher) {
    Arrays.stream(PunishmentType.values())
        .map(PunishCommands::command)
        .forEach(dispatcher::register);
  }

  private static LiteralArgumentBuilder<CommandSourceStack> command(PunishmentType type) {
    return Commands.literal(PREFIX + type.getName())
        .requires(RocketCommands.hasPermissions(type.getPermission().bits()))
        .then(args(type));
  }

  private static ArgumentBuilder<CommandSourceStack, ?> args(PunishmentType type) {
    return Commands.argument("targets", GameProfileArgument.gameProfile())
        .executes(context -> punishPlayers(context.getSource(),
            GameProfileArgument.getGameProfiles(context, "targets"), null, null, type))
        .then(Commands.argument("expiresAt", InstantArgument.instant())
            .executes(context -> punishPlayers(context.getSource(),
                GameProfileArgument.getGameProfiles(context, "targets"),
                context.getArgument("expiresAt", Instant.class), null, type))
            .then(Commands.argument("reason", MessageArgument.message())
                .executes(context -> punishPlayers(context.getSource(),
                    GameProfileArgument.getGameProfiles(context, "targets"),
                    context.getArgument("expiresAt", Instant.class),
                    MessageArgument.getMessage(context, "reason"), type))));
  }

  private static int punishPlayers(CommandSourceStack source, Collection<GameProfile> profiles,
      @Nullable Instant expiresAt, @Nullable Component reason, PunishmentType type)
      throws CommandSyntaxException {
    var punisher = ServerRocketHandle.get(source.getEntity())
        .orElseThrow(RocketCommands.ROCKET_UNAVAILABLE_ERROR::create);
    var gateway = punisher.getGateway();

    Flux.fromIterable(profiles)
        .map(GameProfile::getId)
        .flatMap(gateway::getUser)
        .flatMap(user -> gateway
            .punish(user.id(), punisher.getUserId(), null,
                reason == null ? null : reason.getString(), expiresAt, type)
            .doOnError(error -> source.getServer().execute(() -> source.sendFailure(
                new TranslatableComponent("commands.rocket.punish.failure", error.getMessage()))))
            .onErrorResume(__ -> Mono.empty())
            .doOnSuccess(punishment -> source.getServer().execute(() -> source.sendSuccess(
                new TranslatableComponent("commands.rocket.punish.success", user.displayName(),
                    ComponentUtils.wrapInSquareBrackets(
                        new TextComponent(punishment.id().toHexString())
                            .withStyle(style -> style
                                .withColor(ChatFormatting.GREEN)
                                .withClickEvent(new ClickEvent(ClickEvent.Action.COPY_TO_CLIPBOARD,
                                    punishment.id().toHexString()))
                                .withHoverEvent(new HoverEvent(HoverEvent.Action.SHOW_TEXT,
                                    new TranslatableComponent("chat.copy.click")))
                                .withInsertion(punishment.id().toHexString())))),
                true))))
        .subscribeOn(Schedulers.boundedElastic())
        .subscribe();

    return 1;
  }
}
