package net.rocketpowered.connector.capability;

import java.util.Optional;
import java.util.Set;
import javax.annotation.Nullable;
import org.bson.types.ObjectId;
import it.unimi.dsi.fastutil.objects.ReferenceArraySet;
import net.minecraft.server.level.ServerPlayer;
import net.minecraft.world.entity.Entity;
import net.rocketpowered.common.GameProfile;
import net.rocketpowered.common.Punishment;
import net.rocketpowered.common.Role;
import net.rocketpowered.common.SocialProfile;
import net.rocketpowered.common.util.BitValue;
import net.rocketpowered.connector.server.ServerDist;
import net.rocketpowered.sdk.interf.GameServerInterface;
import net.rocketpowered.sdk.network.protocol.gameserver.GameServerConnectionHandler;
import reactor.core.Disposable;
import reactor.math.MathFlux;

public class ServerRocketHandle extends CommonRocketHandle {

  private ServerPlayer player;

  private final GameServerInterface gateway;

  private final Set<Disposable> listeners = new ReferenceArraySet<>();

  public ServerRocketHandle(ObjectId userId, ServerPlayer player, GameServerInterface gateway) {
    super(userId);
    this.player = player;
    this.gateway = gateway;

    // We can guarantee the implementation of GameServerGateway as it's set by us.
    ((GameServerConnectionHandler) gateway).clearUserCaches(userId);

    this.listeners.add(this.gateway
        .getGameProfileFeed(this.getUserId())
        .doOnNext(this::handleProjectMember)
        .flatMapIterable(GameProfile::roles)
        .flatMap(this.gateway::getRoleFeed)
        .subscribe(this::replaceRole));
    this.listeners.add(this.gateway.getPunishmentFeed(this.getUserId())
        .subscribe(this::handlePunishment));
    this.listeners.add(this.gateway.getSocialProfileFeed(this.getUserId())
        .subscribe(this::handleSocialProfile));
    gateway.getActivePunishments(userId)
        .filter(punishment -> punishment.type().isMute())
        .as(MathFlux::max)
        .subscribe(this::setMute);
  }

  public ServerPlayer getPlayer() {
    return this.player;
  }

  public GameServerInterface getGateway() {
    return this.gateway;
  }

  public void setPlayer(ServerPlayer player) {
    this.player = player;
  }

  private void setMute(@Nullable Punishment mute) {
    if (this.mute != null && this.mute.expiresAfter(mute)) {
      return;
    }
    this.mute = mute;
    this.setChanged();
  }

  private void handlePunishment(Punishment punishment) {
    switch (punishment.type()) {
      case BAN:
        this.player.connection.disconnect(
            ServerDist.createBanMessage(punishment.reason(), punishment.expiresAt()));
        break;
      case MUTE:
        this.setMute(punishment);
        break;
    }
  }

  private void handleSocialProfile(SocialProfile socialProfile) {
    this.socialProfile = socialProfile;
    this.player.refreshDisplayName();
    this.setChanged();
  }

  private void handleProjectMember(GameProfile projectMember) {
    this.roles.clear();
    this.computePermissions();
    this.player.refreshDisplayName();
    this.setChanged();
  }

  private void replaceRole(Role role) {
    if (!this.roles.add(role)) {
      this.roles.remove(role);
      this.roles.add(role);
    }
    this.computePermissions();
    this.player.refreshDisplayName();
    this.setChanged();
  }

  private void computePermissions() {
    this.computedPermissions = this.roles.stream()
        .mapToLong(Role::permissions)
        .reduce((result, element) -> result | element)
        .orElse(BitValue.NILL);
    this.player.getServer().getCommands().sendCommands(this.player);
  }

  public void dispose() {
    this.listeners.forEach(Disposable::dispose);
  }

  public static Optional<ServerRocketHandle> get(Entity entity) {
    return entity.getCapability(RocketHandle.CAPABILITY)
        .filter(ServerRocketHandle.class::isInstance)
        .map(ServerRocketHandle.class::cast);
  }
}
