package net.rocketpowered.connector;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.nio.file.Files;
import java.nio.file.Path;
import org.slf4j.Logger;
import com.mojang.logging.LogUtils;
import net.minecraft.commands.synchronization.ArgumentTypes;
import net.minecraft.commands.synchronization.EmptyArgumentSerializer;
import net.minecraft.world.entity.player.Player;
import net.minecraftforge.common.ForgeConfigSpec;
import net.minecraftforge.common.MinecraftForge;
import net.minecraftforge.common.capabilities.RegisterCapabilitiesEvent;
import net.minecraftforge.event.RegisterCommandsEvent;
import net.minecraftforge.event.entity.living.LivingAttackEvent;
import net.minecraftforge.event.entity.player.PlayerEvent;
import net.minecraftforge.eventbus.api.SubscribeEvent;
import net.minecraftforge.fml.DistExecutor;
import net.minecraftforge.fml.ModLoadingContext;
import net.minecraftforge.fml.common.Mod;
import net.minecraftforge.fml.config.ModConfig;
import net.minecraftforge.fml.event.lifecycle.FMLCommonSetupEvent;
import net.minecraftforge.fml.javafmlmod.FMLJavaModLoadingContext;
import net.minecraftforge.fml.loading.FMLEnvironment;
import net.minecraftforge.fml.loading.FMLPaths;
import net.minecraftforge.fml.loading.JarVersionLookupHandler;
import net.minecraftforge.server.ServerLifecycleHooks;
import net.rocketpowered.connector.capability.RocketHandle;
import net.rocketpowered.connector.client.ClientDist;
import net.rocketpowered.connector.commands.RocketCommands;
import net.rocketpowered.connector.commands.arguments.InstantArgument;
import net.rocketpowered.connector.commands.arguments.ObjectIdArgument;
import net.rocketpowered.connector.network.NetworkManager;
import net.rocketpowered.connector.server.ServerDist;
import net.rocketpowered.sdk.Rocket;
import net.rocketpowered.sdk.network.ConnectionManager;

@Mod(RocketConnector.ID)
public class RocketConnector {

  public static final String ID = "rocketconnector";

  public static final String VERSION =
      JarVersionLookupHandler.getImplementationVersion(RocketConnector.class).orElse("[version]");

  private static final Logger logger = LogUtils.getLogger();

  private static final ForgeConfigSpec configSpec;
  public static final CommonConfig config;

  static {
    var specPair = new ForgeConfigSpec.Builder().configure(CommonConfig::new);
    configSpec = specPair.getRight();
    config = specPair.getLeft();
  }

  /**
   * Singleton.
   */
  private static RocketConnector instance;

  /**
   * Mod distribution.
   */
  private final ModDist modDist;

  private final Path modPath;

  private final ConnectionManager connectionManager;

  public RocketConnector() {
    instance = this;

    this.modPath = FMLPaths.CONFIGDIR.get().resolve(ID);
    if (!Files.exists(this.modPath)) {
      try {
        Files.createDirectory(this.modPath);
      } catch (IOException e) {
        throw new UncheckedIOException(e);
      }
    }

    this.modDist = DistExecutor.safeRunForDist(() -> ClientDist::new, () -> ServerDist::new);

    var modEventBus = FMLJavaModLoadingContext.get().getModEventBus();
    modEventBus.addListener(this::handleCommonSetup);
    modEventBus.addListener(this::handleRegisterCapabilities);

    MinecraftForge.EVENT_BUS.register(this);

    ModLoadingContext.get().registerConfig(ModConfig.Type.COMMON, configSpec);

    NetworkManager.registerMessages();

    this.connectionManager = new ConnectionManager(config.serviceName::get, this.modDist::login);
    Rocket.setConnectionManager(this.connectionManager);
  }

  private void handleCommonSetup(FMLCommonSetupEvent event) {
    ArgumentTypes.register(ID + ":instant", InstantArgument.class,
        new EmptyArgumentSerializer<>(InstantArgument::instant));
    ArgumentTypes.register(ID + ":objectid", ObjectIdArgument.class,
        new EmptyArgumentSerializer<>(ObjectIdArgument::objectId));
  }

  private void handleRegisterCapabilities(RegisterCapabilitiesEvent event) {
    event.register(RocketHandle.class);
  }

  public void init() {
    if (!config.enabled.get()) {
      logger.info("Rocket Connector disabled by config.");
      return;
    }

    if (!this.modDist.setup()) {
      logger.warn("Rocket Connector setup failed, check console for other warnings.");
      return;
    }

    logger.info("Initializing Rocket Connector");
    this.connectionManager.init();
  }

  public ModDist getModDist() {
    return this.modDist;
  }

  public ClientDist getClientDist() {
    if (this.modDist instanceof ClientDist clientDist) {
      return clientDist;
    }
    throw new IllegalStateException("Cannot access client from server dist");
  }

  public ConnectionManager getConnectionManager() {
    return this.connectionManager;
  }

  public Path getModPath() {
    return this.modPath;
  }

  @SubscribeEvent
  public void handleRegisterCommands(RegisterCommandsEvent event) {
    RocketCommands.register(event.getDispatcher(), event.getEnvironment());
  }

  @SubscribeEvent
  public void handleLivingAttack(LivingAttackEvent event) {
    if (event.getSource().getEntity() instanceof Player
        && event.getEntity() instanceof Player) {
      var attacker =
          event.getSource().getEntity().getCapability(RocketHandle.CAPABILITY).orElse(null);
      var attacked =
          event.getEntity().getCapability(RocketHandle.CAPABILITY).orElse(null);
      // As long as one guild optional is present, we can compare them.
      if (attacker != null && attacked != null
          && attacker.getGuild().isPresent()
          && attacker.getGuild().equals(attacked.getGuild())) {
        event.setCanceled(true);
      }
    }
  }

  @SubscribeEvent
  public void handlePlayerNameFormat(PlayerEvent.NameFormat event) {
    event.getPlayer().getCapability(RocketHandle.CAPABILITY)
        .map(handle -> handle.formatDisplayName(event.getDisplayname()))
        .ifPresent(event::setDisplayname);
  }

  public static RocketConnector getInstance() {
    return instance;
  }

  public static boolean isActive() {
    return config.enabled.get() && (!FMLEnvironment.dist.isDedicatedServer()
        || ServerLifecycleHooks.getCurrentServer().usesAuthentication());
  }
}
