/*
 * Crafting Dead
 * Copyright (C) 2022  NexusNode LTD
 *
 * This Non-Commercial Software License Agreement (the "Agreement") is made between
 * you (the "Licensee") and NEXUSNODE (BRAD HUNTER). (the "Licensor").
 * By installing or otherwise using Crafting Dead (the "Software"), you agree to be
 * bound by the terms and conditions of this Agreement as may be revised from time
 * to time at Licensor's sole discretion.
 *
 * If you do not agree to the terms and conditions of this Agreement do not download,
 * copy, reproduce or otherwise use any of the source code available online at any time.
 *
 * https://github.com/nexusnode/crafting-dead/blob/1.18.x/LICENSE.txt
 *
 * https://craftingdead.net/terms.php
 */

package com.craftingdead.protect.whitelist;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.time.Instant;
import java.util.ArrayList;
import java.util.List;
import java.util.jar.JarFile;
import java.util.jar.Manifest;
import java.util.stream.Stream;
import com.craftingdead.protect.client.integrity.util.HashUtil;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;

/**
 * Generates the allowed-mods.json whitelist by scanning a directory of mod JARs.
 * This is designed to be called during the release build process.
 */
public final class AllowedModsGenerator {
  
  private static final Gson GSON = new GsonBuilder()
      .setPrettyPrinting()
      .create();
  
  private AllowedModsGenerator() {}
  
  /**
   * Main entry point for Gradle task execution.
   * 
   * @param args [0] = mods directory path, [1] = output JSON path
   */
  public static void main(String[] args) {
    if (args.length < 2) {
      System.err.println("Usage: AllowedModsGenerator <modsDir> <outputJsonPath>");
      System.exit(1);
    }
    
    Path modsDir = Paths.get(args[0]);
    Path outputPath = Paths.get(args[1]);
    
    try {
      System.out.println("Generating allowed-mods whitelist...");
      System.out.println("Scanning directory: " + modsDir.toAbsolutePath());
      
      AllowedModsData data = generateWhitelist(modsDir);
      
      System.out.println("Found " + data.getMods().size() + " mods");
      
      writeWhitelist(data, outputPath);
      
      System.out.println("Whitelist written to: " + outputPath.toAbsolutePath());
      System.out.println("Generation complete!");
      
    } catch (Exception e) {
      System.err.println("Failed to generate whitelist: " + e.getMessage());
      e.printStackTrace();
      System.exit(1);
    }
  }
  
  /**
   * Generates the whitelist by scanning all JAR files in the given directory.
   * 
   * @param modsDir Directory containing mod JAR files
   * @return AllowedModsData containing all discovered mods
   */
  public static AllowedModsData generateWhitelist(Path modsDir) throws IOException {
    if (!Files.exists(modsDir) || !Files.isDirectory(modsDir)) {
      throw new IOException("Mods directory does not exist: " + modsDir);
    }
    
    AllowedModsData data = new AllowedModsData();
    data.setGeneratedAt(Instant.now().toString());
    
    List<AllowedModEntry> mods = new ArrayList<>();
    
    try (Stream<Path> files = Files.list(modsDir)) {
      files.filter(Files::isRegularFile)
           .filter(p -> p.toString().toLowerCase().endsWith(".jar"))
           .forEach(jarPath -> {
             try {
               AllowedModEntry entry = scanModJar(jarPath);
               if (entry != null) {
                 mods.add(entry);
                 System.out.println("  ✓ " + entry.getModId() + " " + entry.getVersion());
               }
             } catch (Exception e) {
               System.err.println("  ✗ Failed to scan: " + jarPath.getFileName() + " - " + e.getMessage());
             }
           });
    }
    
    data.setMods(mods);
    return data;
  }
  
  /**
   * Scans a single mod JAR file and extracts metadata.
   * 
   * @param jarPath Path to the JAR file
   * @return AllowedModEntry with mod information, or null if not a valid mod
   */
  private static AllowedModEntry scanModJar(Path jarPath) throws IOException {
    String filename = jarPath.getFileName().toString();
    
    // Compute SHA-256 hash
    String sha256 = HashUtil.sha256(jarPath);
    
    // Try to extract mod metadata
    String modId = null;
    String version = null;
    
    try (JarFile jarFile = new JarFile(jarPath.toFile())) {
      // Try to read mods.toml (Forge 1.18.2 format)
      var modsToml = jarFile.getEntry("META-INF/mods.toml");
      if (modsToml != null) {
        try (InputStream is = jarFile.getInputStream(modsToml)) {
          String content = new String(is.readAllBytes());
          modId = extractModIdFromToml(content);
          version = extractVersionFromToml(content);
        }
      }
      
      // Fallback: try manifest
      if (modId == null) {
        Manifest manifest = jarFile.getManifest();
        if (manifest != null) {
          var attrs = manifest.getMainAttributes();
          modId = attrs.getValue("Implementation-Title");
          version = attrs.getValue("Implementation-Version");
        }
      }
      
      // Fallback: use filename
      if (modId == null) {
        modId = extractModIdFromFilename(filename);
      }
      if (version == null) {
        version = extractVersionFromFilename(filename);
      }
    }
    
    if (modId == null) {
      return null;
    }
    
    return new AllowedModEntry(modId, version != null ? version : "unknown", filename, sha256);
  }
  
  /**
   * Extracts modId from mods.toml content.
   */
  private static String extractModIdFromToml(String content) {
    String[] lines = content.split("\n");
    for (String line : lines) {
      line = line.trim();
      if (line.startsWith("modId") || line.startsWith("modId=")) {
        int equalsIdx = line.indexOf('=');
        if (equalsIdx != -1) {
          String value = line.substring(equalsIdx + 1).trim();
          return value.replace("\"", "").replace("'", "");
        }
      }
    }
    return null;
  }
  
  /**
   * Extracts version from mods.toml content.
   */
  private static String extractVersionFromToml(String content) {
    String[] lines = content.split("\n");
    for (String line : lines) {
      line = line.trim();
      if (line.startsWith("version") && !line.startsWith("versionRange")) {
        int equalsIdx = line.indexOf('=');
        if (equalsIdx != -1) {
          String value = line.substring(equalsIdx + 1).trim();
          return value.replace("\"", "").replace("'", "").replace("${file.jarVersion}", "");
        }
      }
    }
    return null;
  }
  
  /**
   * Attempts to extract mod ID from filename.
   * Example: "crafting-dead-core-1.18.2-1.8.0.147-all.jar" -> "craftingdeadcore"
   */
  private static String extractModIdFromFilename(String filename) {
    String name = filename.replace(".jar", "");
    // Remove version patterns
    name = name.replaceAll("-\\d+\\.\\d+.*", "");
    name = name.replaceAll("-all$", "");
    name = name.replace("-", "");
    return name.toLowerCase();
  }
  
  /**
   * Attempts to extract version from filename.
   */
  private static String extractVersionFromFilename(String filename) {
    // Pattern: x.x.x.x or x.x.x
    String[] parts = filename.split("-");
    for (String part : parts) {
      if (part.matches("\\d+\\.\\d+.*")) {
        return part.replace(".jar", "");
      }
    }
    return "unknown";
  }
  
  /**
   * Writes the whitelist data to a JSON file.
   * 
   * @param data The whitelist data
   * @param outputPath Output file path
   */
  public static void writeWhitelist(AllowedModsData data, Path outputPath) throws IOException {
    Files.createDirectories(outputPath.getParent());
    String json = GSON.toJson(data);
    Files.writeString(outputPath, json);
  }
}
