/*
 * Crafting Dead
 * Copyright (C) 2022  NexusNode LTD
 *
 * This Non-Commercial Software License Agreement (the "Agreement") is made between
 * you (the "Licensee") and NEXUSNODE (BRAD HUNTER). (the "Licensor").
 * By installing or otherwise using Crafting Dead (the "Software"), you agree to be
 * bound by the terms and conditions of this Agreement as may be revised from time
 * to time at Licensor's sole discretion.
 *
 * If you do not agree to the terms and conditions of this Agreement do not download,
 * copy, reproduce or otherwise use any of the source code available online at any time.
 *
 * https://github.com/nexusnode/crafting-dead/blob/1.18.x/LICENSE.txt
 *
 * https://craftingdead.net/terms.php
 */

package com.craftingdead.protect.whitelist;

import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.Gson;

/**
 * Loads and provides access to the allowed mods whitelist.
 * Supports both config-based override and built-in fallback.
 */
public final class AllowedMods {
  
  private static final Logger logger = LoggerFactory.getLogger(AllowedMods.class);
  private static final Gson GSON = new Gson();
  
  private static final String BUILTIN_WHITELIST_PATH = "/assets/cdprotect/whitelist/allowed-mods.json";
  
  // Use configDir parameter in getInstance() instead of hardcoded path
  private static Path configWhitelistPath;
  
  private static AllowedMods instance;
  
  private final Map<String, AllowedModEntry> whitelist = new HashMap<>();
  private final boolean loaded;
  
  private AllowedMods(Path configDir) {
    boolean success = false;
    
    // Calculate config whitelist path
    if (configDir != null) {
      configWhitelistPath = configDir.resolve("cdprotect").resolve("allowed-mods.json");
      
      // Try to load from config first (override)
      if (Files.exists(configWhitelistPath)) {
        logger.info("Loading allowed mods whitelist from config: {}", configWhitelistPath);
        success = loadFromFile(configWhitelistPath);
      }
    }
    
    // Fallback to built-in resource
    if (!success) {
      logger.info("Loading allowed mods whitelist from built-in resource");
      success = loadFromResource();
    }
    
    this.loaded = success;
    
    if (success) {
      logger.info("Allowed mods whitelist loaded: {} entries", whitelist.size());
    } else {
      logger.warn("Failed to load allowed mods whitelist - validation will be limited");
    }
  }
  
  /**
   * Gets the singleton instance with a specific config directory.
   * 
   * @param configDir The config directory path (e.g., FMLPaths.CONFIGDIR.get())
   */
  public static synchronized AllowedMods getInstance(Path configDir) {
    if (instance == null) {
      instance = new AllowedMods(configDir);
    }
    return instance;
  }
  
  /**
   * Gets the singleton instance (uses cached instance if already initialized).
   */
  public static synchronized AllowedMods getInstance() {
    if (instance == null) {
      // Initialize with null configDir - will only use built-in resource
      instance = new AllowedMods(null);
    }
    return instance;
  }
  
  /**
   * Loads whitelist from a file.
   */
  private boolean loadFromFile(Path path) {
    try {
      String json = Files.readString(path);
      AllowedModsData data = GSON.fromJson(json, AllowedModsData.class);
      populateWhitelist(data);
      return true;
    } catch (IOException e) {
      logger.error("Failed to load whitelist from file: {}", path, e);
      return false;
    }
  }
  
  /**
   * Loads whitelist from built-in resource.
   */
  private boolean loadFromResource() {
    try (InputStream is = getClass().getResourceAsStream(BUILTIN_WHITELIST_PATH)) {
      if (is == null) {
        logger.warn("Built-in whitelist resource not found: {}", BUILTIN_WHITELIST_PATH);
        return false;
      }
      
      try (InputStreamReader reader = new InputStreamReader(is)) {
        AllowedModsData data = GSON.fromJson(reader, AllowedModsData.class);
        populateWhitelist(data);
        return true;
      }
    } catch (IOException e) {
      logger.error("Failed to load whitelist from resource", e);
      return false;
    }
  }
  
  /**
   * Populates the whitelist map from data.
   */
  private void populateWhitelist(AllowedModsData data) {
    whitelist.clear();
    if (data != null && data.getMods() != null) {
      for (AllowedModEntry entry : data.getMods()) {
        whitelist.put(entry.getModId(), entry);
      }
    }
  }
  
  /**
   * Gets whitelisted mod information by mod ID.
   * 
   * @param modId The mod ID to look up
   * @return Optional containing the whitelist entry if found
   */
  public Optional<AllowedModEntry> getWhitelistedInfo(String modId) {
    return Optional.ofNullable(whitelist.get(modId));
  }
  
  /**
   * Checks if a mod is allowed based on its ID, version, and hash.
   * 
   * @param modId Mod ID
   * @param version Mod version
   * @param hash SHA-256 hash of the mod JAR
   * @return true if the mod is whitelisted and matches, false otherwise
   */
  public boolean isModAllowed(String modId, String version, String hash) {
    AllowedModEntry entry = whitelist.get(modId);
    if (entry == null) {
      return false;
    }
    
    // Check version match (allow "unknown" as wildcard)
    boolean versionMatch = entry.getVersion().equals("unknown") 
        || entry.getVersion().equals(version);
    
    // Check hash match
    boolean hashMatch = entry.getSha256().equalsIgnoreCase(hash);
    
    return versionMatch && hashMatch;
  }
  
  /**
   * Checks if the whitelist was successfully loaded.
   */
  public boolean isLoaded() {
    return loaded;
  }
  
  /**
   * Gets the number of whitelisted mods.
   */
  public int getWhitelistSize() {
    return whitelist.size();
  }
  
  /**
   * Checks if a mod ID exists in the whitelist (regardless of version/hash).
   */
  public boolean isModIdWhitelisted(String modId) {
    return whitelist.containsKey(modId);
  }
}
