/*
 * Crafting Dead
 * Copyright (C) 2022  NexusNode LTD
 *
 * This Non-Commercial Software License Agreement (the "Agreement") is made between
 * you (the "Licensee") and NEXUSNODE (BRAD HUNTER). (the "Licensor").
 * By installing or otherwise using Crafting Dead (the "Software"), you agree to be
 * bound by the terms and conditions of this Agreement as may be revised from time
 * to time at Licensor's sole discretion.
 *
 * If you do not agree to the terms and conditions of this Agreement do not download,
 * copy, reproduce or otherwise use any of the source code available online at any time.
 *
 * https://github.com/nexusnode/crafting-dead/blob/1.18.x/LICENSE.txt
 *
 * https://craftingdead.net/terms.php
 */

package com.craftingdead.protect.client.integrity.util;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

/**
 * Utility class for file hashing operations.
 */
public final class HashUtil {
  
  private static final int BUFFER_SIZE = 8192;
  private static final char[] HEX_ARRAY = "0123456789abcdef".toCharArray();
  
  private HashUtil() {}
  
  /**
   * Computes SHA-256 hash of a file.
   * 
   * @param path Path to the file to hash
   * @return Hexadecimal string representation of the hash
   * @throws IOException If file cannot be read
   */
  public static String sha256(Path path) throws IOException {
    try {
      MessageDigest digest = MessageDigest.getInstance("SHA-256");
      
      try (InputStream is = Files.newInputStream(path)) {
        byte[] buffer = new byte[BUFFER_SIZE];
        int read;
        while ((read = is.read(buffer)) != -1) {
          digest.update(buffer, 0, read);
        }
      }
      
      return bytesToHex(digest.digest());
    } catch (NoSuchAlgorithmException e) {
      throw new RuntimeException("SHA-256 algorithm not available", e);
    }
  }
  
  /**
   * Computes SHA-256 hash of an input stream.
   * Does not close the stream.
   * 
   * @param is InputStream to hash
   * @return Hexadecimal string representation of the hash
   * @throws IOException If stream cannot be read
   */
  public static String sha256(InputStream is) throws IOException {
    try {
      MessageDigest digest = MessageDigest.getInstance("SHA-256");
      
      byte[] buffer = new byte[BUFFER_SIZE];
      int read;
      while ((read = is.read(buffer)) != -1) {
        digest.update(buffer, 0, read);
      }
      
      return bytesToHex(digest.digest());
    } catch (NoSuchAlgorithmException e) {
      throw new RuntimeException("SHA-256 algorithm not available", e);
    }
  }
  
  /**
   * Converts byte array to hexadecimal string.
   */
  private static String bytesToHex(byte[] bytes) {
    char[] hexChars = new char[bytes.length * 2];
    for (int i = 0; i < bytes.length; i++) {
      int v = bytes[i] & 0xFF;
      hexChars[i * 2] = HEX_ARRAY[v >>> 4];
      hexChars[i * 2 + 1] = HEX_ARRAY[v & 0x0F];
    }
    return new String(hexChars);
  }
}
