/*
 * Crafting Dead
 * Copyright (C) 2022  NexusNode LTD
 *
 * This Non-Commercial Software License Agreement (the "Agreement") is made between
 * you (the "Licensee") and NEXUSNODE (BRAD HUNTER). (the "Licensor").
 * By installing or otherwise using Crafting Dead (the "Software"), you agree to be
 * bound by the terms and conditions of this Agreement as may be revised from time
 * to time at Licensor's sole discretion.
 *
 * If you do not agree to the terms and conditions of this Agreement do not download,
 * copy, reproduce or otherwise use any of the source code available online at any time.
 *
 * https://github.com/nexusnode/crafting-dead/blob/1.18.x/LICENSE.txt
 *
 * https://craftingdead.net/terms.php
 */

package com.craftingdead.protect.client.integrity.scanner;

import java.util.ArrayList;
import java.util.HashSet;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.craftingdead.protect.client.integrity.LoadedModInfo;
import com.craftingdead.protect.client.integrity.ModValidationResult;
import com.craftingdead.protect.client.integrity.config.ModListConfig;
import com.craftingdead.protect.whitelist.AllowedModEntry;
import com.craftingdead.protect.whitelist.AllowedMods;

/**
 * Validates loaded mods against whitelist and blacklist configurations.
 * Now integrates with the automated whitelist generator system.
 */
public final class ModValidator {
  
  private static final Logger logger = LoggerFactory.getLogger(ModValidator.class);
  
  private ModValidator() {}
  
  /**
   * Validates loaded mods against the configuration.
   * Now also validates against the automated whitelist.
   * 
   * @param loadedMods List of loaded mods
   * @param config Mod list configuration
   * @return Validation result
   */
  public static ModValidationResult validate(List<LoadedModInfo> loadedMods, 
      ModListConfig config) {
    
    Set<String> loadedModIds = new HashSet<>();
    for (LoadedModInfo mod : loadedMods) {
      loadedModIds.add(mod.modId());
    }
    
    // Check for missing required mods
    List<String> missingRequired = new ArrayList<>();
    for (String requiredMod : config.getRequiredMods()) {
      if (!loadedModIds.contains(requiredMod)) {
        missingRequired.add(requiredMod);
        logger.warn("Missing required mod: {}", requiredMod);
      }
    }
    
    // Check for blacklisted mods
    List<String> blacklisted = new ArrayList<>();
    Set<String> blacklistedSet = config.getBlacklistedModsSet();
    for (String modId : loadedModIds) {
      if (blacklistedSet.contains(modId)) {
        blacklisted.add(modId);
        logger.warn("Detected blacklisted mod: {}", modId);
      }
    }
    
    // Check for unknown mods in strict mode OR against whitelist
    List<String> unknown = new ArrayList<>();
    AllowedMods whitelist = AllowedMods.getInstance();
    
    if (config.isStrictMode() || whitelist.isLoaded()) {
      Set<String> allowedSet = config.getAllowedModsSet();
      Set<String> requiredSet = config.getRequiredModsSet();
      
      for (LoadedModInfo mod : loadedMods) {
        String modId = mod.modId();
        
        // Skip vanilla/forge mods
        if (isVanillaOrForgeMod(modId)) {
          continue;
        }
        
        // Check against manual allowed list
        boolean manuallyAllowed = allowedSet.contains(modId) || requiredSet.contains(modId);
        
        // Check against automated whitelist
        boolean whitelisted = false;
        if (whitelist.isLoaded()) {
          Optional<AllowedModEntry> entry = whitelist.getWhitelistedInfo(modId);
          if (entry.isPresent()) {
            AllowedModEntry allowedEntry = entry.get();
            boolean hashMatch = allowedEntry.getSha256().equalsIgnoreCase(mod.sha256());
            
            if (hashMatch) {
              whitelisted = true;
            } else {
              logger.warn("Mod '{}' hash mismatch! Expected: {}, Got: {}", 
                  modId, allowedEntry.getSha256(), mod.sha256());
              unknown.add(modId + " (modified)");
            }
          }
        }
        
        // Report if not allowed by either method
        if (!manuallyAllowed && !whitelisted) {
          if (config.isStrictMode()) {
            unknown.add(modId);
            logger.info("Unknown mod detected: {}", modId);
          } else if (whitelist.isLoaded()) {
            unknown.add(modId + " (not in whitelist)");
            logger.warn("Mod not in whitelist: {}", modId);
          }
        }
      }
    }
    
    return new ModValidationResult(missingRequired, blacklisted, unknown);
  }
  
  /**
   * Checks if a mod ID is a vanilla or forge system mod.
   */
  private static boolean isVanillaOrForgeMod(String modId) {
    return modId.equals("minecraft") 
        || modId.equals("forge")
        || modId.startsWith("forge");
  }
}
