/*
 * Crafting Dead
 * Copyright (C) 2022  NexusNode LTD
 *
 * This Non-Commercial Software License Agreement (the "Agreement") is made between
 * you (the "Licensee") and NEXUSNODE (BRAD HUNTER). (the "Licensor").
 * By installing or otherwise using Crafting Dead (the "Software"), you agree to be
 * bound by the terms and conditions of this Agreement as may be revised from time
 * to time at Licensor's sole discretion.
 *
 * If you do not agree to the terms and conditions of this Agreement do not download,
 * copy, reproduce or otherwise use any of the source code available online at any time.
 *
 * https://github.com/nexusnode/crafting-dead/blob/1.18.x/LICENSE.txt
 *
 * https://craftingdead.net/terms.php
 */

package com.craftingdead.protect.client.integrity.scanner;

import java.net.URL;
import java.security.CodeSource;
import java.security.ProtectionDomain;
import java.util.ArrayList;
import java.util.List;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.craftingdead.protect.client.integrity.ClassIntegrityResult;

/**
 * Scans critical classes to ensure they're loaded from expected sources.
 */
public final class ClassIntegrityScanner {
  
  private static final Logger logger = LoggerFactory.getLogger(ClassIntegrityScanner.class);
  
  // Critical Crafting Dead classes to verify
  private static final String[] CRITICAL_CLASSES = {
      "com.craftingdead.core.CraftingDead",
      "com.craftingdead.immerse.CraftingDeadImmerse",
      "com.craftingdead.survival.CraftingDeadSurvival"
  };
  
  private ClassIntegrityScanner() {}
  
  /**
   * Scans critical classes to verify their source locations.
   * 
   * @return List of class integrity results
   */
  public static List<ClassIntegrityResult> scanClassIntegrity() {
    List<ClassIntegrityResult> results = new ArrayList<>();
    
    for (String className : CRITICAL_CLASSES) {
      try {
        ClassIntegrityResult result = checkClassIntegrity(className);
        results.add(result);
        
        if (!result.valid()) {
          logger.warn("Class integrity violation: {} loaded from {} instead of {}",
              className, result.actualSource(), result.expectedSource());
        }
      } catch (ClassNotFoundException e) {
        logger.warn("Critical class not found: {}", className);
        results.add(new ClassIntegrityResult(className, "unknown", "not_found", false));
      } catch (Exception e) {
        logger.error("Failed to check integrity for class: {}", className, e);
      }
    }
    
    return results;
  }
  
  /**
   * Checks the integrity of a specific class.
   */
  private static ClassIntegrityResult checkClassIntegrity(String className) 
      throws ClassNotFoundException {
    
    Class<?> clazz = Class.forName(className);
    
    // Get the protection domain and code source
    ProtectionDomain pd = clazz.getProtectionDomain();
    if (pd == null) {
      return new ClassIntegrityResult(className, "unknown", "no_protection_domain", false);
    }
    
    CodeSource cs = pd.getCodeSource();
    if (cs == null) {
      return new ClassIntegrityResult(className, "unknown", "no_code_source", false);
    }
    
    URL location = cs.getLocation();
    String actualSource = location != null ? location.toString() : "unknown";
    
    // Determine expected source based on class package
    String expectedSource = determineExpectedSource(className);
    
    // Check if actual matches expected
    boolean valid = actualSource.contains(expectedSource);
    
    return new ClassIntegrityResult(className, expectedSource, actualSource, valid);
  }
  
  /**
   * Determines the expected JAR source for a class based on its package.
   */
  private static String determineExpectedSource(String className) {
    if (className.startsWith("com.craftingdead.core")) {
      return "crafting-dead-core";
    } else if (className.startsWith("com.craftingdead.immerse")) {
      return "crafting-dead-immerse";
    } else if (className.startsWith("com.craftingdead.survival")) {
      return "crafting-dead-survival";
    } else if (className.startsWith("com.craftingdead.decoration")) {
      return "crafting-dead-decoration";
    }
    return "crafting-dead";
  }
}
