/*
 * Crafting Dead
 * Copyright (C) 2022  NexusNode LTD
 *
 * This Non-Commercial Software License Agreement (the "Agreement") is made between
 * you (the "Licensee") and NEXUSNODE (BRAD HUNTER). (the "Licensor").
 * By installing or otherwise using Crafting Dead (the "Software"), you agree to be
 * bound by the terms and conditions of this Agreement as may be revised from time
 * to time at Licensor's sole discretion.
 *
 * If you do not agree to the terms and conditions of this Agreement do not download,
 * copy, reproduce or otherwise use any of the source code available online at any time.
 *
 * https://github.com/nexusnode/crafting-dead/blob/1.18.x/LICENSE.txt
 *
 * https://craftingdead.net/terms.php
 */

package com.craftingdead.protect.client.integrity.config;

import java.io.IOException;
import java.nio.file.Files;
import java.nio.file.Path;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;

/**
 * Loads mod list configurations from JSON files.
 */
public final class ConfigLoader {
  
  private static final Logger logger = LoggerFactory.getLogger(ConfigLoader.class);
  private static final Gson GSON = new GsonBuilder()
      .setPrettyPrinting()
      .create();
  
  private ConfigLoader() {}
  
  /**
   * Loads mod list configuration from a JSON file.
   * Creates a default configuration if the file doesn't exist.
   */
  public static ModListConfig loadModListConfig(Path configPath) {
    if (!Files.exists(configPath)) {
      logger.info("Config file not found: {}, creating default", configPath);
      ModListConfig defaultConfig = createDefaultConfig();
      saveModListConfig(configPath, defaultConfig);
      return defaultConfig;
    }
    
    try {
      String json = Files.readString(configPath);
      return GSON.fromJson(json, ModListConfig.class);
    } catch (IOException e) {
      logger.error("Failed to load config from: {}", configPath, e);
      return createDefaultConfig();
    }
  }
  
  /**
   * Saves mod list configuration to a JSON file.
   */
  public static void saveModListConfig(Path configPath, ModListConfig config) {
    try {
      Files.createDirectories(configPath.getParent());
      String json = GSON.toJson(config);
      Files.writeString(configPath, json);
    } catch (IOException e) {
      logger.error("Failed to save config to: {}", configPath, e);
    }
  }
  
  /**
   * Creates a default mod list configuration.
   */
  private static ModListConfig createDefaultConfig() {
    ModListConfig config = new ModListConfig();
    config.setRequiredMods(java.util.List.of(
        "minecraft",
        "forge",
        "craftingdead",
        "craftingdeadimmerse",
        "craftingdeadsurvival"
    ));
    config.setBlacklistedMods(java.util.List.of(
        "wurst",
        "baritone",
        "meteor-client",
        "lambda",
        "aristois"
    ));
    config.setStrictMode(false);
    return config;
  }
}
