package com.craftingdead.protect;

/**
 * Represents the protection mode for the application.
 * 
 * DEV mode: Minimal protection, allows debugging and profiling.
 * RELEASE mode: Full protection, stricter enforcement of security checks.
 * 
 * The mode is controlled via the JVM property: -Dcd.protectionMode=dev|release
 */
public enum ProtectionMode {
    
    /**
     * Development mode - minimal protection, allows debugging.
     */
    DEV,
    
    /**
     * Release mode - full protection with strict enforcement.
     */
    RELEASE;
    
    private static final String PROPERTY_NAME = "cd.protectionMode";
    private static final ProtectionMode CURRENT_MODE = determineMode();
    
    /**
     * Gets the current protection mode based on system property.
     * 
     * @return The current protection mode (defaults to DEV if not specified)
     */
    public static ProtectionMode current() {
        return CURRENT_MODE;
    }
    
    /**
     * Checks if the current mode is DEV.
     * 
     * @return true if running in development mode
     */
    public static boolean isDev() {
        return CURRENT_MODE == DEV;
    }
    
    /**
     * Checks if the current mode is RELEASE.
     * 
     * @return true if running in release mode
     */
    public static boolean isRelease() {
        return CURRENT_MODE == RELEASE;
    }
    
    /**
     * Determines the protection mode from system property.
     * Defaults to DEV if property is not set or invalid.
     */
    private static ProtectionMode determineMode() {
        try {
            String mode = System.getProperty(PROPERTY_NAME, "dev").toLowerCase();
            
            switch (mode) {
                case "release":
                case "prod":
                case "production":
                    return RELEASE;
                case "dev":
                case "development":
                case "debug":
                default:
                    return DEV;
            }
        } catch (Exception e) {
            // If we can't read system property, default to DEV for safety
            return DEV;
        }
    }
    
    /**
     * Gets a human-readable description of this mode.
     * 
     * @return Description string
     */
    public String getDescription() {
        switch (this) {
            case DEV:
                return "Development Mode (debugging enabled)";
            case RELEASE:
                return "Release Mode (full protection)";
            default:
                return "Unknown Mode";
        }
    }
    
    @Override
    public String toString() {
        return name().toLowerCase();
    }
}
