package com.craftingdead.protect;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * JNI wrapper for native protection layer (cdprotect library).
 * 
 * This class provides access to native (C/C++) implementations of protection checks
 * that are harder to reverse-engineer and tamper with than pure Java code.
 * 
 * CURSEFORGE-SAFE GUARANTEES:
 * - Native library operates ONLY on this JVM process (no system-wide hooks)
 * - NO automatic downloads or updates
 * - NO network calls from native code
 * - NO filesystem modifications (except temp extraction if needed)
 * - NO scanning of other processes
 * - NO data collection or telemetry
 * - NO kernel-level hooks or rootkit behavior
 * 
 * The native layer ONLY provides:
 * - Enhanced integrity checking (harder to bypass than pure Java)
 * - Enhanced environment detection (debuggers, agents within our process)
 * - Challenge-response verification (anti-tamper)
 * 
 * GRACEFUL DEGRADATION:
 * If the native library fails to load, all methods return safe defaults and
 * the protection system falls back to pure Java implementations.
 */
public final class NativeProtection {
    
    private static final Logger LOGGER = LoggerFactory.getLogger(NativeProtection.class);
    
    /**
     * Whether the native library was successfully loaded.
     */
    private static final boolean LOADED;
    
    /**
     * Library name without platform-specific prefix/suffix.
     * Platform-specific variations:
     * - Windows: cdprotect.dll
     * - Linux: libcdprotect.so
     * - macOS: libcdprotect.dylib
     */
    private static final String LIBRARY_NAME = "cdprotect";
    
    static {
        boolean loaded = false;
        try {
            // Try to load native library directly (assuming it's already loaded by anticheat module)
            // This will succeed if the anticheat module has already loaded it
            try {
                int testResult = nativeChallenge(42);
                boolean envTest = nativeEnvClean();
                loaded = true;
                LOGGER.info("Native Protection: LOADED (challenge test: {}, env test: {})", testResult, envTest);
            } catch (UnsatisfiedLinkError linkError) {
                // Native library not yet loaded - this is expected when cd-protect is used standalone
                LOGGER.debug("Native library not loaded yet (will be loaded by anticheat module)");
            }
            
        } catch (UnsatisfiedLinkError e) {
            LOGGER.warn("Native protection library '{}' could not be loaded: {}", 
                    LIBRARY_NAME, e.getMessage());
            LOGGER.debug("Native library load error details", e);
        } catch (Throwable t) {
            LOGGER.error("Unexpected error checking native protection library", t);
        }
        LOADED = loaded;
    }
    
    private NativeProtection() {
        throw new UnsupportedOperationException("Utility class");
    }
    
    /**
     * Checks if the native protection library was successfully loaded.
     * 
     * @return true if native library is available, false otherwise
     */
    public static boolean isLoaded() {
        return LOADED;
    }
    
    /**
     * Performs native-level integrity verification.
     * 
     * This method verifies the integrity of the JAR file containing this class
     * using native code that is harder to tamper with than pure Java.
     * 
     * Implementation details (native side):
     * - Computes SHA-256 hash of the JAR file
     * - Compares against expected hash (baked into native library)
     * - Returns false if tampering is detected
     * - Returns false on any error (safe default)
     * 
     * CURSEFORGE-SAFE:
     * - Only checks our own JAR file
     * - NO network calls
     * - NO filesystem modifications
     * - NO scanning of other files
     * 
     * @return true if integrity check passes, false if tampering detected or library not loaded
     */
    public static native boolean nativeIntegrityOk();
    
    /**
     * Performs native-level environment cleanliness check.
     * 
     * This method detects suspicious debugging/profiling environments using
     * native code that is harder to bypass than pure Java checks.
     * 
     * Implementation details (native side):
     * - Checks for debugger attachment (within this process only)
     * - Detects known profiling agents (via JVM args inspection)
     * - Looks for suspicious JNI/JVMTI agents
     * - Returns false if hostile environment detected
     * - Returns false on any error (safe default)
     * 
     * CURSEFORGE-SAFE:
     * - Only inspects THIS JVM process
     * - NO scanning of other processes
     * - NO system-wide hooks
     * - NO kernel-level checks
     * 
     * @return true if environment appears clean, false if suspicious or library not loaded
     */
    public static native boolean nativeEnvClean();
    
    /**
     * Performs a challenge-response computation for anti-tamper verification.
     * 
     * This method computes a non-trivial response based on the input seed,
     * using native code that is harder to reverse-engineer. The Java side
     * can verify the response to ensure the native library hasn't been replaced.
     * 
     * Implementation details (native side):
     * - Performs a deterministic computation on the seed
     * - Uses obfuscated algorithm (not cryptographically secure, just hard to RE)
     * - Returns consistent results for same seed
     * - Returns 0 on any error
     * 
     * Usage pattern:
     * <pre>
     * int seed = new Random().nextInt();
     * int nativeResult = NativeProtection.nativeChallenge(seed);
     * int expectedResult = computeExpectedChallenge(seed); // Java implementation
     * if (nativeResult != expectedResult) {
     *     // Native library may have been tampered with
     * }
     * </pre>
     * 
     * CURSEFORGE-SAFE:
     * - Pure computation, no side effects
     * - NO network calls
     * - NO filesystem access
     * - NO system modifications
     * 
     * @param seed Input value for the challenge
     * @return Computed response value, or 0 if library not loaded
     */
    public static native int nativeChallenge(int seed);
    
    /**
     * Safe wrapper for nativeIntegrityOk() that handles library not loaded.
     * 
     * @return true if native library is loaded AND integrity check passes, false otherwise
     */
    public static boolean checkIntegrity() {
        if (!LOADED) {
            return false;
        }
        try {
            return nativeIntegrityOk();
        } catch (Throwable t) {
            LOGGER.error("Native integrity check failed with exception", t);
            return false;
        }
    }
    
    /**
     * Safe wrapper for nativeEnvClean() that handles library not loaded.
     * 
     * @return true if native library is loaded AND environment is clean, false otherwise
     */
    public static boolean checkEnvironment() {
        if (!LOADED) {
            return false;
        }
        try {
            return nativeEnvClean();
        } catch (Throwable t) {
            LOGGER.error("Native environment check failed with exception", t);
            return false;
        }
    }
    
    /**
     * Safe wrapper for nativeChallenge() that handles library not loaded.
     * 
     * @param seed Input seed for challenge
     * @return Challenge response, or 0 if library not loaded or error occurs
     */
    public static int performChallenge(int seed) {
        if (!LOADED) {
            return 0;
        }
        try {
            return nativeChallenge(seed);
        } catch (Throwable t) {
            LOGGER.error("Native challenge failed with exception", t);
            return 0;
        }
    }
    
    /**
     * Computes the expected response for a challenge seed (Java-side verification).
     * 
     * This should match the native implementation's algorithm. It's intentionally
     * simpler than the native version (which may use more obfuscation).
     * 
     * This is used to verify the native library hasn't been replaced with a stub.
     * 
     * @param seed Input seed
     * @return Expected response value
     */
    public static int computeExpectedChallenge(int seed) {
        // Simple algorithm that matches native implementation
        // Native side uses the same logic but in C/C++ (harder to reverse)
        int result = seed;
        result = (result * 0x9E3779B9) ^ (result >>> 16);
        result = (result * 0x85EBCA6B) ^ (result >>> 13);
        result = (result * 0xC2B2AE35) ^ (result >>> 16);
        return result;
    }
    
    /**
     * Verifies that the native library is genuine by performing a challenge-response check.
     * 
     * @param seed Random seed for the challenge
     * @return true if native library responds correctly, false otherwise
     */
    public static boolean verifyNativeLibrary(int seed) {
        if (!LOADED) {
            return false;
        }
        
        int nativeResult = performChallenge(seed);
        int expectedResult = computeExpectedChallenge(seed);
        
        if (nativeResult != expectedResult) {
            LOGGER.warn("Native library challenge verification FAILED! " +
                    "Expected: {}, Got: {}. Native library may be compromised.", 
                    expectedResult, nativeResult);
            return false;
        }
        
        return true;
    }
    
    /**
     * Gets a status string describing the native protection state.
     * Useful for logging and diagnostics.
     * 
     * @return Human-readable status string
     */
    public static String getStatus() {
        if (!LOADED) {
            return "Native protection: UNAVAILABLE (using Java-only fallback)";
        }
        
        StringBuilder status = new StringBuilder("Native protection: ACTIVE");
        
        try {
            boolean integrity = nativeIntegrityOk();
            boolean environment = nativeEnvClean();
            
            status.append(" [Integrity: ").append(integrity ? "OK" : "FAIL");
            status.append(", Environment: ").append(environment ? "CLEAN" : "SUSPICIOUS");
            status.append("]");
        } catch (Throwable t) {
            status.append(" [Error: ").append(t.getMessage()).append("]");
        }
        
        return status.toString();
    }
}
