package com.craftingdead.protect;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.io.IOException;
import java.io.InputStream;
import java.net.URL;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.ProtectionDomain;
import java.util.HexFormat;

/**
 * Provides runtime integrity checking capabilities.
 * Computes and verifies SHA-256 hashes of JAR files and resources.
 * 
 * Now enhanced with native-level integrity checks when available.
 * Falls back to pure Java implementation if native library is not loaded.
 * 
 * All methods return booleans or null values instead of throwing exceptions
 * to ensure protection checks never crash the application.
 */
public final class IntegrityChecker {
    
    private static final Logger LOGGER = LoggerFactory.getLogger(IntegrityChecker.class);
    private static final String HASH_ALGORITHM = "SHA-256";
    private static final int BUFFER_SIZE = 8192;
    
    private IntegrityChecker() {
        throw new UnsupportedOperationException("Utility class");
    }
    
    /**
     * Computes the SHA-256 hash of the JAR file containing the given class.
     * 
     * @param clazz The class whose containing JAR will be hashed
     * @return Hex-encoded SHA-256 hash, or null if computation fails
     */
    public static String computeJarHash(Class<?> clazz) {
        if (clazz == null) {
            return null;
        }
        
        try {
            ProtectionDomain domain = clazz.getProtectionDomain();
            if (domain == null || domain.getCodeSource() == null) {
                return null;
            }
            
            URL location = domain.getCodeSource().getLocation();
            if (location == null) {
                return null;
            }
            
            return computeUrlHash(location);
        } catch (Exception e) {
            // Silent failure - never crash on integrity check
            return null;
        }
    }
    
    /**
     * Computes the SHA-256 hash of a resource inside the JAR.
     * 
     * @param resourcePath The path to the resource (e.g., "META-INF/MANIFEST.MF")
     * @param classLoader The ClassLoader to use for loading the resource
     * @return Hex-encoded SHA-256 hash, or null if computation fails
     */
    public static String computeResourceHash(String resourcePath, ClassLoader classLoader) {
        if (resourcePath == null || classLoader == null) {
            return null;
        }
        
        try {
            URL resourceUrl = classLoader.getResource(resourcePath);
            if (resourceUrl == null) {
                return null;
            }
            
            return computeUrlHash(resourceUrl);
        } catch (Exception e) {
            // Silent failure
            return null;
        }
    }
    
    /**
     * Computes the SHA-256 hash of content at the given URL.
     * 
     * @param url The URL to hash
     * @return Hex-encoded SHA-256 hash, or null if computation fails
     */
    private static String computeUrlHash(URL url) {
        if (url == null) {
            return null;
        }
        
        try (InputStream input = url.openStream()) {
            MessageDigest digest = MessageDigest.getInstance(HASH_ALGORITHM);
            byte[] buffer = new byte[BUFFER_SIZE];
            int bytesRead;
            
            while ((bytesRead = input.read(buffer)) != -1) {
                digest.update(buffer, 0, bytesRead);
            }
            
            byte[] hashBytes = digest.digest();
            return HexFormat.of().formatHex(hashBytes);
        } catch (IOException | NoSuchAlgorithmException e) {
            // Silent failure
            return null;
        }
    }
    
    /**
     * Compares two hash strings for equality.
     * 
     * @param hash1 First hash
     * @param hash2 Second hash
     * @return true if both hashes are non-null and equal (case-insensitive), false otherwise
     */
    public static boolean compareHashes(String hash1, String hash2) {
        if (hash1 == null || hash2 == null) {
            return false;
        }
        return hash1.equalsIgnoreCase(hash2);
    }
    
    /**
     * Verifies the integrity of a class's containing JAR against an expected hash.
     * 
     * TODO: In release mode, this should trigger stricter enforcement if verification fails.
     * 
     * @param clazz The class to verify
     * @param expectedHash The expected SHA-256 hash (hex-encoded)
     * @return true if the hash matches, false otherwise
     */
    public static boolean verifyJarIntegrity(Class<?> clazz, String expectedHash) {
        String actualHash = computeJarHash(clazz);
        return compareHashes(actualHash, expectedHash);
    }
    
    /**
     * Verifies the integrity of a resource against an expected hash.
     * 
     * TODO: In release mode, this should trigger stricter enforcement if verification fails.
     * 
     * @param resourcePath The path to the resource
     * @param classLoader The ClassLoader to use
     * @param expectedHash The expected SHA-256 hash (hex-encoded)
     * @return true if the hash matches, false otherwise
     */
    public static boolean verifyResourceIntegrity(String resourcePath, ClassLoader classLoader, String expectedHash) {
        String actualHash = computeResourceHash(resourcePath, classLoader);
        return compareHashes(actualHash, expectedHash);
    }
    
    /**
     * Performs comprehensive integrity check using both Java and native methods.
     * 
     * In RELEASE mode with native library loaded:
     * - Checks both Java SHA-256 hash AND native integrity verification
     * - Both must pass for overall integrity to be confirmed
     * 
     * In DEV mode or without native library:
     * - Only performs Java-based checks
     * - Logs advisory warnings but doesn't fail strictly
     * 
     * @param clazz The class to verify
     * @param expectedHash The expected SHA-256 hash (can be null in dev mode)
     * @return true if integrity checks pass, false otherwise
     */
    public static boolean performComprehensiveIntegrityCheck(Class<?> clazz, String expectedHash) {
        ProtectionMode mode = ProtectionMode.current();
        boolean javaIntegrityOk = true;
        boolean nativeIntegrityOk = true;
        
        // Java-based integrity check
        if (expectedHash != null) {
            javaIntegrityOk = verifyJarIntegrity(clazz, expectedHash);
            if (!javaIntegrityOk) {
                LOGGER.warn("Java integrity check FAILED for {}", clazz.getName());
            }
        }
        
        // NOTE: Native integrity checks are handled by crafting-dead-anticheat module
        // cd-protect is a pure Java library with no native calls
        
        // In release mode, Java checks must pass
        if (ProtectionMode.isRelease()) {
            return javaIntegrityOk;
        }
        
        // In dev mode, just report but don't fail
        if (!javaIntegrityOk) {
            LOGGER.debug("Integrity check failures in DEV mode (not enforced)");
        }
        
        return true; // Dev mode always passes
    }
    
    /**
     * Quick integrity check that only uses native layer if available.
     * Faster than comprehensive check, suitable for frequent validation.
     * 
     * @return true if integrity appears OK, false if suspicious
     */
    public static boolean quickIntegrityCheck() {
        // NOTE: Native integrity checks are handled by crafting-dead-anticheat module
        // cd-protect uses pure Java checks only (return true to avoid false positives)
        return true;
    }
}
