package com.craftingdead.protect;

import java.lang.management.ManagementFactory;
import java.lang.management.RuntimeMXBean;
import java.util.List;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Detects debugging and profiling environments at runtime.
 * Enhanced with native-code protection layer for deeper environment checks.
 * 
 * All methods return booleans and never throw exceptions to ensure
 * protection checks don't crash the application.
 */
public final class EnvironmentChecker {
    
    private static final Logger LOGGER = LoggerFactory.getLogger(EnvironmentChecker.class);
    
    private static final String JDWP_OPTION = "jdwp";
    private static final String AGENT_OPTION = "-javaagent";
    
    // Known debugger/profiler signatures
    private static final String[] SUSPICIOUS_AGENTS = {
        "intellij",
        "eclipse",
        "netbeans",
        "jrebel",
        "yourkit",
        "jprofiler",
        "visualvm",
        "jdwp",
        "debug"
    };
    
    private EnvironmentChecker() {
        throw new UnsupportedOperationException("Utility class");
    }
    
    /**
     * Checks if JDWP (Java Debug Wire Protocol) is active.
     * This indicates that a debugger is or can be attached.
     * 
     * @return true if JDWP debugging is detected, false otherwise
     */
    public static boolean isDebuggingActive() {
        try {
            RuntimeMXBean runtimeMxBean = ManagementFactory.getRuntimeMXBean();
            List<String> arguments = runtimeMxBean.getInputArguments();
            
            for (String arg : arguments) {
                if (arg.toLowerCase().contains(JDWP_OPTION)) {
                    return true;
                }
            }
            
            return false;
        } catch (Exception e) {
            // Silent failure - assume no debugging if we can't check
            return false;
        }
    }
    
    /**
     * Checks if any Java agents are attached to the JVM.
     * Agents can be used for debugging, profiling, or code manipulation.
     * 
     * @return true if any -javaagent option is detected, false otherwise
     */
    public static boolean hasJavaAgents() {
        try {
            RuntimeMXBean runtimeMxBean = ManagementFactory.getRuntimeMXBean();
            List<String> arguments = runtimeMxBean.getInputArguments();
            
            for (String arg : arguments) {
                if (arg.startsWith(AGENT_OPTION)) {
                    return true;
                }
            }
            
            return false;
        } catch (Exception e) {
            // Silent failure
            return false;
        }
    }
    
    /**
     * Checks for known debugger or profiler signatures in JVM arguments.
     * This includes IntelliJ IDEA, Eclipse, JProfiler, YourKit, etc.
     * 
     * @return true if suspicious agent signatures are detected, false otherwise
     */
    public static boolean hasSuspiciousAgents() {
        try {
            RuntimeMXBean runtimeMxBean = ManagementFactory.getRuntimeMXBean();
            List<String> arguments = runtimeMxBean.getInputArguments();
            
            for (String arg : arguments) {
                String lowerArg = arg.toLowerCase();
                for (String suspicious : SUSPICIOUS_AGENTS) {
                    if (lowerArg.contains(suspicious)) {
                        return true;
                    }
                }
            }
            
            return false;
        } catch (Exception e) {
            // Silent failure
            return false;
        }
    }
    
    /**
     * Performs a comprehensive check for any debugging or profiling environment.
     * This combines all detection methods including native-level checks when available.
     * 
     * In RELEASE mode with native library loaded:
     * - Checks both Java-based detection AND native environment verification
     * - Both layers must report clean for overall environment to be considered safe
     * 
     * In DEV mode or without native library:
     * - Only performs Java-based checks
     * - Logs advisory warnings but doesn't fail strictly
     * 
     * @return true if any debugging/profiling is detected, false otherwise
     */
    public static boolean isHostileEnvironment() {
        ProtectionMode mode = ProtectionMode.current();
        boolean javaHostile = isDebuggingActive() || hasJavaAgents() || hasSuspiciousAgents();
        
        // NOTE: Native environment checking is handled by crafting-dead-anticheat module
        // cd-protect is a pure Java library with no native calls
        
        if (javaHostile) {
            LOGGER.warn("Java-based hostile environment detection triggered");
        }
        
        return javaHostile;
    }
    
    /**
     * Performs comprehensive environment verification using both Java and native methods.
     * This is more thorough than isHostileEnvironment() and includes challenge-response.
     * 
     * In RELEASE mode with native library loaded:
     * - Performs Java checks, native environment check, AND native challenge-response
     * - All checks must pass for environment to be considered clean
     * 
     * In DEV mode or without native library:
     * - Only performs Java-based checks
     * - Logs advisory warnings
     * 
     * @return true if environment is clean, false if hostile/suspicious
     */
    public static boolean performComprehensiveEnvironmentCheck() {
        ProtectionMode mode = ProtectionMode.current();
        
        // Java-based checks
        boolean javaClean = !isHostileEnvironment();
        
        // NOTE: Native checks and challenge-response are handled by crafting-dead-anticheat module
        // cd-protect is a pure Java library with no native calls
        
        // In release mode, Java checks must pass
        if (ProtectionMode.isRelease()) {
            return javaClean;
        }
        
        // In dev mode, just report but don't fail
        if (!javaClean) {
            LOGGER.debug("Environment check failures in DEV mode (not enforced)");
        }
        
        return true; // Dev mode always passes
    }
    
    /**
     * Quick environment check that only uses native layer if available.
     * Faster than comprehensive check, suitable for frequent validation.
     * 
     * @return true if environment appears clean, false if suspicious
     */
    public static boolean quickEnvironmentCheck() {
        // NOTE: Native protection is handled by crafting-dead-anticheat module
        // cd-protect uses pure Java checks only
        return !isHostileEnvironment();
    }
    
    /**
     * Gets all JVM input arguments as a single string for logging/debugging.
     * Should only be used in development mode.
     * 
     * @return Space-separated JVM arguments, or empty string if unavailable
     */
    public static String getJvmArguments() {
        try {
            RuntimeMXBean runtimeMxBean = ManagementFactory.getRuntimeMXBean();
            return String.join(" ", runtimeMxBean.getInputArguments());
        } catch (Exception e) {
            return "";
        }
    }
}
