package io.scalecube.services.transport.jackson;

import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.lang.reflect.Type;
import java.util.Collections;
import java.util.Map;
import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.MapperFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.fasterxml.jackson.databind.json.JsonMapper;
import com.fasterxml.jackson.datatype.jsr310.JavaTimeModule;
import io.scalecube.services.transport.api.DataCodec;
import io.scalecube.services.transport.api.HeadersCodec;

public final class JacksonCodec implements DataCodec, HeadersCodec {

  public static final String CONTENT_TYPE = "application/json";

  private final ObjectMapper mapper;

  public JacksonCodec() {
    this(JsonMapper.builder()
        .configure(MapperFeature.ALLOW_FINAL_FIELDS_AS_MUTATORS, false)
        .configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false)
        .configure(SerializationFeature.FAIL_ON_EMPTY_BEANS, false)
        .configure(DeserializationFeature.READ_UNKNOWN_ENUM_VALUES_AS_NULL, true)
        .configure(SerializationFeature.WRITE_DATES_AS_TIMESTAMPS, false)
        .visibility(PropertyAccessor.ALL, JsonAutoDetect.Visibility.ANY)
        .serializationInclusion(JsonInclude.Include.NON_NULL)
        .configure(SerializationFeature.WRITE_ENUMS_USING_TO_STRING, true)
        .addModule(new JavaTimeModule())
        .build());
  }

  public JacksonCodec(ObjectMapper mapper) {
    this.mapper = mapper;
  }

  @Override
  public String contentType() {
    return CONTENT_TYPE;
  }

  @Override
  public void encode(OutputStream stream, Map<String, String> headers) throws IOException {
    mapper.writeValue(stream, headers);
  }

  @Override
  public void encode(OutputStream stream, Object value) throws IOException {
    mapper.writeValue(stream, value);
  }

  @Override
  public Map<String, String> decode(InputStream stream) throws IOException {
    return stream.available() == 0
        ? Collections.emptyMap()
        : mapper.readValue(stream, new TypeReference<>() {});
  }

  @Override
  public Object decode(InputStream stream, Type type) throws IOException {
    return mapper.readValue(stream, mapper.getTypeFactory().constructType(type));
  }
}
