package io.scalecube.services.routing;

import java.lang.reflect.InvocationTargetException;
import java.util.concurrent.ConcurrentHashMap;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class Routers {
  private static final Logger LOGGER = LoggerFactory.getLogger(Routers.class);

  private static final ConcurrentHashMap<Class<? extends Router>, Router> routers =
      new ConcurrentHashMap<>();

  private Routers() {}

  /**
   * Get router instance by a given router class. The class should have a default constructor.
   * Otherwise no router can be created
   *
   * @param routerType the type of the Router.
   * @return instance of the Router.
   */
  public static Router getRouter(Class<? extends Router> routerType) {
    return routers.computeIfAbsent(routerType, Routers::create);
  }

  private static Router create(Class<? extends Router> routerType) {
    try {
      return routerType.getConstructor().newInstance();
    } catch (Throwable t) {
      if (t instanceof InvocationTargetException) {
        t = t.getCause();
      }
      LOGGER.error("Create router type: {} failed: {}", routerType, t);
      throw t instanceof RuntimeException runtimeException
          ? runtimeException
          : new RuntimeException(t);
    }
  }
}
