package io.scalecube.services.discovery.api;

import java.util.Objects;
import io.scalecube.net.Address;
import io.scalecube.services.discovery.api.ServiceDiscoveryEvent.Type;
import io.scalecube.services.registry.api.ServiceRegistry;
import reactor.core.publisher.Flux;
import reactor.core.scheduler.Scheduler;

public record ServiceDiscoveryContext(
    String id,
    Address address,
    ServiceDiscovery discovery,
    ServiceRegistry serviceRegistry,
    Scheduler scheduler) {

  public static Builder builder() {
    return new Builder();
  }

  /**
   * Returns copy builder.
   *
   * @param other instance to copy from
   * @return new builder instance
   */
  public static Builder from(ServiceDiscoveryContext other) {
    return new Builder()
        .id(other.id)
        .address(other.address)
        .discovery(other.discovery)
        .serviceRegistry(other.serviceRegistry)
        .scheduler(other.scheduler);
  }


  /**
   * Returns stream of service discovery events. Can be called before or after
   * {@link ServiceDiscovery#start()}. If it's called before then new events will be streamed, if
   * it's called after then {@link ServiceRegistry#listServiceEndpoints()} will be turned to service
   * discovery events of type {@link Type#ENDPOINT_ADDED}, and concateneted with a stream of live
   * events.
   *
   * @return stream of service discovery events
   */
  public Flux<ServiceDiscoveryEvent> listen() {
    return Flux.fromStream(serviceRegistry.listServiceEndpoints().stream())
        .map(ServiceDiscoveryEvent::newEndpointAdded)
        .concatWith(discovery.listen())
        .subscribeOn(scheduler)
        .publishOn(scheduler);
  }

  public static class Builder {

    private String id;
    private Address address;
    private ServiceDiscovery discovery;
    private ServiceRegistry serviceRegistry;
    private Scheduler scheduler;

    private Builder() {}

    public Builder id(String id) {
      this.id = id;
      return this;
    }

    public Builder address(Address address) {
      this.address = address;
      return this;
    }

    public Builder discovery(ServiceDiscovery discovery) {
      this.discovery = discovery;
      return this;
    }

    public Builder serviceRegistry(ServiceRegistry serviceRegistry) {
      this.serviceRegistry = serviceRegistry;
      return this;
    }

    public Builder scheduler(Scheduler scheduler) {
      this.scheduler = scheduler;
      return this;
    }

    public ServiceDiscoveryContext build() {
      return new ServiceDiscoveryContext(
          Objects.requireNonNull(this.id, "id"),
          Objects.requireNonNull(this.address, "address"),
          Objects.requireNonNull(this.discovery, "discovery"),
          Objects.requireNonNull(this.serviceRegistry, "serviceRegistry"),
          Objects.requireNonNull(this.scheduler, "scheduler"));
    }
  }
}
