package io.scalecube.services;

import java.util.HashMap;
import java.util.Map;
import java.util.Set;
import io.scalecube.net.Address;
import io.scalecube.services.api.Qualifier;

/**
 * Service reference. This is merge of service method information together with service registration
 * and with service endpoint.
 */
public record ServiceReference(
    String qualifier,
    String endpointId,
    String namespace,
    Set<String> contentTypes,
    Map<String, String> tags,
    String action,
    Address address,
    boolean secured) {

  public ServiceReference(
      String qualifier,
      String endpointId,
      String namespace,
      Set<String> contentTypes,
      Map<String, String> tags,
      String action,
      Address address,
      boolean secured) {
    this.qualifier = qualifier;
    this.endpointId = endpointId;
    this.namespace = namespace;
    this.contentTypes = Set.copyOf(contentTypes);
    this.tags = Map.copyOf(tags);
    this.action = action;
    this.address = address;
    this.secured = secured;
  }

  /**
   * Constructor for service reference.
   *
   * @param serviceMethodDefinition service method info
   * @param serviceRegistration service registration
   * @param serviceEndpoint service node info
   */
  public ServiceReference(
      ServiceMethodDefinition serviceMethodDefinition,
      ServiceRegistration serviceRegistration,
      ServiceEndpoint serviceEndpoint) {
    this(
        Qualifier.asString(serviceRegistration.namespace(), serviceMethodDefinition.action()),
        serviceEndpoint.id(),
        serviceRegistration.namespace(),
        serviceEndpoint.contentTypes(),
        mergeTags(serviceMethodDefinition, serviceRegistration, serviceEndpoint),
        serviceMethodDefinition.action(),
        serviceEndpoint.address(),
        serviceMethodDefinition.secured());
  }

  public String oldQualifier() {
    return Qualifier.DELIMITER + this.qualifier();
  }

  private static Map<String, String> mergeTags(
      ServiceMethodDefinition serviceMethodDefinition,
      ServiceRegistration serviceRegistration,
      ServiceEndpoint serviceEndpoint) {
    Map<String, String> tags = new HashMap<>();
    tags.putAll(serviceEndpoint.tags());
    tags.putAll(serviceRegistration.tags());
    tags.putAll(serviceMethodDefinition.tags());
    return tags;
  }
}
