package io.scalecube.services;

import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.StringJoiner;
import io.scalecube.services.auth.Authenticator;
import io.scalecube.services.auth.PrincipalMapper;
import io.scalecube.services.exceptions.ServiceProviderErrorMapper;
import io.scalecube.services.transport.api.ServiceMessageDataDecoder;

public record ServiceInfo(
    Object serviceInstance,
    Map<String, String> tags,
    ServiceProviderErrorMapper errorMapper,
    ServiceMessageDataDecoder dataDecoder,
    Authenticator<Object> authenticator,
    PrincipalMapper<Object, Object> principalMapper) {

  public ServiceInfo(
      Object serviceInstance,
      Map<String, String> tags,
      ServiceProviderErrorMapper errorMapper,
      ServiceMessageDataDecoder dataDecoder,
      Authenticator<Object> authenticator,
      PrincipalMapper<Object, Object> principalMapper) {
    this.serviceInstance = serviceInstance;
    this.tags = Map.copyOf(tags);
    this.errorMapper = errorMapper;
    this.dataDecoder = dataDecoder;
    this.authenticator = authenticator;
    this.principalMapper = principalMapper;
  }

  public static Builder from(ServiceInfo serviceInfo) {
    return new Builder(serviceInfo);
  }

  public static Builder fromServiceInstance(Object serviceInstance) {
    return new Builder(serviceInstance);
  }

  @Override
  public String toString() {
    return new StringJoiner(", ", ServiceInfo.class.getSimpleName() + "[", "]")
        .add("serviceInstance=" + serviceInstance)
        .add("tags(" + tags.size() + ")")
        .add("errorMapper=" + errorMapper)
        .add("dataDecoder=" + dataDecoder)
        .add("authenticator=" + authenticator)
        .add("principalMapper=" + principalMapper)
        .toString();
  }

  public static class Builder {

    private final Object serviceInstance;
    private final Map<String, String> tags = new HashMap<>();
    private ServiceProviderErrorMapper errorMapper;
    private ServiceMessageDataDecoder dataDecoder;
    private Authenticator<Object> authenticator;
    private PrincipalMapper<Object, Object> principalMapper;

    private Builder(ServiceInfo serviceInfo) {
      this.serviceInstance = serviceInfo.serviceInstance;
      this.tags.putAll(new HashMap<>(serviceInfo.tags));
      this.errorMapper = serviceInfo.errorMapper;
      this.dataDecoder = serviceInfo.dataDecoder;
      this.authenticator = serviceInfo.authenticator;
      this.principalMapper = serviceInfo.principalMapper;
    }

    private Builder(Object serviceInstance) {
      this.serviceInstance = serviceInstance;
    }

    /**
     * Setter for {@code tags}. Merges this {@code tags} with {@code Microservices.tags}. If keys
     * are clashing this {@code tags} shall override {@code Microservices.tags}.
     *
     * @param key tag key; not null
     * @param value tag value; not null
     * @return this builder
     */
    public Builder tag(String key, String value) {
      Objects.requireNonNull(key, "tag key");
      Objects.requireNonNull(value, "tag value");
      tags.put(key, value);
      return this;
    }

    /**
     * Setter for {@code errorMapper}. Overrides default {@code Microservices.errorMapper}.
     *
     * @param errorMapper error mapper; not null
     * @return this buidler
     */
    public Builder errorMapper(ServiceProviderErrorMapper errorMapper) {
      this.errorMapper = Objects.requireNonNull(errorMapper, "errorMapper");
      return this;
    }

    /**
     * Setter for {@code dataDecoder}. Overrides default {@code Microservices.dataDecoder}.
     *
     * @param dataDecoder data decoder; not null
     * @return this builder
     */
    public Builder dataDecoder(ServiceMessageDataDecoder dataDecoder) {
      this.dataDecoder = Objects.requireNonNull(dataDecoder, "dataDecoder");
      return this;
    }

    /**
     * Setter for {@code authenticator}. Overrides default {@code Microservices.authenticator}.
     *
     * @param authenticator authenticator; optional
     * @param <T> type of auth data returned by authenticator
     * @return this builder
     */
    @SuppressWarnings("unchecked")
    public <T> Builder authenticator(Authenticator<? extends T> authenticator) {
      // noinspection unchecked
      this.authenticator = (Authenticator<Object>) authenticator;
      return this;
    }

    /**
     * Setter for {@code principalMapper}. Overrides default {@code Microservices.principalMapper}.
     *
     * @param principalMapper principalMapper; optional
     * @param <T> auth data type
     * @param <R> principal type
     * @return this builder
     */
    @SuppressWarnings("unchecked")
    public <T, R> Builder principalMapper(PrincipalMapper<? super T, ? extends R> principalMapper) {
      // noinspection unchecked
      this.principalMapper = (PrincipalMapper<Object, Object>) principalMapper;
      return this;
    }

    public Builder errorMapperIfAbsent(ServiceProviderErrorMapper errorMapper) {
      if (this.errorMapper == null) {
        this.errorMapper = errorMapper;
      }
      return this;
    }

    public Builder dataDecoderIfAbsent(ServiceMessageDataDecoder dataDecoder) {
      if (this.dataDecoder == null) {
        this.dataDecoder = dataDecoder;
      }
      return this;
    }

    public Builder authenticatorIfAbsent(Authenticator<Object> authenticator) {
      if (this.authenticator == null) {
        this.authenticator = authenticator;
      }
      return this;
    }

    public Builder principalMapperIfAbsent(PrincipalMapper<Object, Object> principalMapper) {
      if (this.principalMapper == null) {
        this.principalMapper = principalMapper;
      }
      return this;
    }

    public ServiceInfo build() {
      return new ServiceInfo(
          this.serviceInstance,
          this.tags,
          this.errorMapper,
          this.dataDecoder,
          this.authenticator,
          this.principalMapper);
    }
  }
}
