package io.scalecube.services;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.StringJoiner;
import java.util.stream.Collectors;
import io.scalecube.net.Address;

public record ServiceEndpoint(
    String id,
    Address address,
    Set<String> contentTypes,
    Map<String, String> tags,
    Collection<ServiceRegistration> serviceRegistrations) implements Serializable {

  public ServiceEndpoint(
      String id,
      Address address,
      Set<String> contentTypes,
      Map<String, String> tags,
      Collection<ServiceRegistration> serviceRegistrations) {
    this.id = id;
    this.address = address;
    this.contentTypes = Set.copyOf(contentTypes);
    this.tags = Map.copyOf(tags);
    this.serviceRegistrations = List.copyOf(serviceRegistrations);
  }

  public static Builder builder() {
    return new Builder();
  }

  public static Builder from(ServiceEndpoint serviceEndpoint) {
    return new Builder(serviceEndpoint);
  }

  /**
   * Creates collection of service references from {@code serviceRegistrations}.
   *
   * @return {@link ServiceReference} collection
   */
  public Collection<ServiceReference> serviceReferences() {
    return serviceRegistrations.stream()
        .flatMap(sr -> sr.methods().stream().map(sm -> new ServiceReference(sm, sr, this)))
        .collect(Collectors.toList());
  }

  @Override
  public String toString() {
    return new StringJoiner(", ", ServiceEndpoint.class.getSimpleName() + "[", "]")
        .add("id=" + id)
        .add("address=" + address)
        .add("contentTypes=" + contentTypes)
        .add("tags=" + tags)
        .add("serviceRegistrations(" + serviceRegistrations.size() + ")")
        .toString();
  }

  public static class Builder {

    private String id;
    private Address address = Address.NULL_ADDRESS;
    private Set<String> contentTypes = Collections.emptySet();
    private Map<String, String> tags = Collections.emptyMap();
    private Collection<ServiceRegistration> serviceRegistrations = new ArrayList<>();

    private Builder() {}

    private Builder(ServiceEndpoint other) {
      this.id = other.id;
      this.address = other.address;
      this.contentTypes = new HashSet<>(other.contentTypes);
      this.tags = new HashMap<>(other.tags);
      this.serviceRegistrations = new ArrayList<>(other.serviceRegistrations);
    }

    public Builder id(String id) {
      this.id = Objects.requireNonNull(id, "id");
      return this;
    }

    public Builder address(Address address) {
      this.address = Objects.requireNonNull(address, "address");
      return this;
    }

    public Builder contentTypes(Set<String> contentTypes) {
      this.contentTypes = new HashSet<>(Objects.requireNonNull(contentTypes, "contentTypes"));
      return this;
    }

    public Builder tags(Map<String, String> tags) {
      this.tags = new HashMap<>(Objects.requireNonNull(tags, "tags"));
      return this;
    }

    /**
     * Adds {@code serviceRegistrations} to collection of {@code serviceRegistrations}.
     *
     * @param serviceRegistrations serviceRegistrations
     * @return this builder
     */
    public Builder appendServiceRegistrations(
        Collection<ServiceRegistration> serviceRegistrations) {
      this.serviceRegistrations.addAll(
          Objects.requireNonNull(serviceRegistrations, "serviceRegistrations"));
      return this;
    }

    /**
     * Setter for {@code serviceRegistrations}.
     *
     * @param serviceRegistrations serviceRegistrations
     * @return this builder
     */
    public Builder serviceRegistrations(Collection<ServiceRegistration> serviceRegistrations) {
      this.serviceRegistrations =
          new ArrayList<>(Objects.requireNonNull(serviceRegistrations, "serviceRegistrations"));
      return this;
    }

    public ServiceEndpoint build() {
      return new ServiceEndpoint(
          Objects.requireNonNull(this.id, "ServiceEndpoint.id is required"),
          Objects.requireNonNull(this.address, "ServiceEndpoint.address is required"),
          this.contentTypes,
          this.tags,
          this.serviceRegistrations);
    }
  }
}
