package flare.eventbus;

import java.util.Set;
import it.unimi.dsi.fastutil.objects.ReferenceOpenHashSet;

final class HashEventDispatcher<E> implements EventDispatcher<E> {

  private final Set<EventSubscriber<E>> subscribers = new ReferenceOpenHashSet<>();
  @SuppressWarnings("unchecked")
  private EventSubscriber<E>[] sortedSubscribers = new EventSubscriber[0];

  @Override
  public boolean register(EventSubscriber<E> subscriber) {
    if (this.subscribers.add(subscriber)) {
      this.sortSubscribers();
      return true;
    }
    return false;
  }

  @Override
  public boolean unregister(EventSubscriber<E> subscriber) {
    if (this.subscribers.remove(subscriber)) {
      this.sortSubscribers();
      return true;
    }
    return false;
  }

  @Override
  public boolean hasSubscribers() {
    return !this.subscribers.isEmpty();
  }

  @SuppressWarnings("unchecked")
  private void sortSubscribers() {
    this.sortedSubscribers = this.subscribers.stream()
        .sorted()
        .toArray(EventSubscriber[]::new);
  }

  @Override
  public boolean dispatch(E event, EventContext context) {
    var changed = false;
    for (int i = 0; i < this.sortedSubscribers.length; i++) {
      changed |= this.dispatchTo(event, context, this.sortedSubscribers[i]);
    }
    if (changed) {
      this.sortSubscribers();
    }
    return changed;
  }

  private boolean dispatchTo(E event, EventContext context, EventSubscriber<E> subscriber) {
    return switch (subscriber.handler().handle(event, context)) {
      case EXPIRED -> this.subscribers.remove(subscriber);
      default -> false;
    };
  }
}
