package flare.eventbus;

import java.util.Optional;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import org.jetbrains.annotations.Nullable;

/**
 * An event bus which dispatches events to subscribed handlers.
 * 
 * @param <T> - root event type
 */
public sealed interface EventBus<T> permits EventBusImpl {

  static EventBus<Object> create() {
    return create(Object.class);
  }

  static <T> EventBus<T> create(Class<T> eventType) {
    return create(eventType, true);
  }

  static <T> EventBus<T> create(Class<T> eventType, boolean hierarchicalDispatch) {
    return create(eventType,
        hierarchicalDispatch ? ClassHierarchyService.cached() : null);
  }

  static <T> EventBus<T> create(Class<T> eventType,
      @Nullable ClassHierarchyService hierarchyService) {
    return new EventBusImpl<>(eventType, hierarchyService);
  }

  /**
   * Publishes an event.
   * 
   * @param <E> - published event type
   * @param event - event to publish
   */
  default <E extends T> void publish(E event) {
    this.publish(event, EventContext.empty());
  }

  /**
   * Publishes a cancellable event.
   * 
   * @param <E> - published event type
   * @param event - event to publish
   * @return {@code true} if the event was cancelled
   */
  default <E extends T> boolean publishCancellable(E event) {
    return this.publish(event, EventContext.cancellable()).isCancelled();
  }

  <E extends T> EventContext publish(E event, EventContext context);

  Class<T> eventType();

  Optional<EventBus<? super T>> parent();

  Set<EventBus<? extends T>> children();

  <E extends T> EventBus<T> addChild(EventBus<E> child);

  <E extends T> EventBus<T> removeChild(EventBus<E> child);

  /**
   * Subscribes all method annotated with {@link Subscribe} declared in the specified object.
   * Passing a {@link Class} to this method will subscribe static methods, passing an instance will
   * subscribe instance methods.
   * 
   * @param object - an object instance or a class
   * @return the event bus
   */
  EventBus<T> subscribeAnnotated(Object object);

  /**
   * Unsubscribes all methods annotated with {@link Subscribe} declared in the specified object.
   * Passing a {@link Class} to this method will unsubscribe static methods, passing an instance
   * will unsubscribe instance methods.
   * 
   * @param object - an object instance or a class
   * @return the event bus
   */
  EventBus<T> unsubscribeAnnotated(Object object);

  default <E extends T> EventBus<T> subscribe(Class<E> eventType, Consumer<E> handler) {
    return this.subscribe(eventType, EventHandler.of(handler));
  }

  default <E extends T> EventBus<T> subscribe(
      Class<E> eventType, BiConsumer<E, EventContext> handler) {
    return this.subscribe(eventType, EventHandler.of(handler));
  }

  default <E extends T> EventBus<T> subscribe(Class<E> eventType, EventHandler<E> handler) {
    return this.subscribe(new EventSubscriber<>(eventType, handler));
  }

  <E extends T> EventBus<T> subscribe(EventSubscriber<E> subscriber);

  <E extends T> EventBus<T> unsubscribe(EventSubscriber<E> subscriber);
}
