package flare.eventbus;

import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BiConsumer;
import java.util.function.Predicate;
import org.jetbrains.annotations.Nullable;

final class BuiltEventHandler<T> implements EventHandler<T> {

  private final boolean ignoreCancelled;
  @Nullable
  private final AtomicInteger expirationCount;
  @Nullable
  private final Predicate<T> expireWhen;
  @Nullable
  private final Predicate<T>[] filters;

  private final BiConsumer<T, EventContext> handler;

  BuiltEventHandler(
      boolean ignoreCancelled,
      @Nullable AtomicInteger expirationCount,
      @Nullable Predicate<T> expireWhen,
      @Nullable Predicate<T>[] filters,
      BiConsumer<T, EventContext> handler) {
    this.ignoreCancelled = ignoreCancelled;
    this.expirationCount = expirationCount;
    this.expireWhen = expireWhen;
    this.filters = filters;
    this.handler = handler;
  }

  @Override
  public Result handle(T event, EventContext context) {
    // Event cancellation
    if (this.ignoreCancelled && context.isCancelled()) {
      return Result.INVALID;
    }

    // Expiration predicate
    if (this.expireWhen != null && this.expireWhen.test(event)) {
      return Result.EXPIRED;
    }

    // Filtering
    if (this.filters != null) {
      for (int i = 0; i < this.filters.length; i++) {
        if (!this.filters[i].test(event)) {
          // Cancelled
          return Result.INVALID;
        }
      }
    }

    this.handler.accept(event, context);

    // Expiration count
    if (this.expirationCount != null && this.expirationCount.decrementAndGet() == 0) {
      return Result.EXPIRED;
    }

    return Result.SUCCESS;
  }
}
