package flare.eventbus;

import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.Arrays;
import java.util.Optional;
import java.util.function.BiConsumer;
import java.util.stream.Stream;
import org.jetbrains.annotations.Nullable;
import com.esotericsoftware.reflectasm.MethodAccess;
import flare.commons.ReflectionUtil;

final class AnnotatedSubscriberUtil {

  private AnnotatedSubscriberUtil() {}

  static <T> Stream<EventSubscriber<? extends T>> findAnnotatedSubscribers(
      Object obj, Class<T> superEventType) {
    return obj.getClass() == Class.class
        ? findStaticAnnotatedSubscribers((Class<?>) obj, superEventType)
        : findInstanceAnnotatedSubscribers(obj, superEventType);
  }

  private static <T> Stream<EventSubscriber<? extends T>> findStaticAnnotatedSubscribers(
      Class<?> clazz, Class<T> superEventType) {
    return Arrays.stream(clazz.getMethods())
        .filter(method -> Modifier.isStatic(method.getModifiers()))
        .filter(method -> method.isAnnotationPresent(Subscribe.class))
        .map(method -> createSubscriber(null, method, method, superEventType));
  }

  private static <T> Stream<EventSubscriber<? extends T>> findInstanceAnnotatedSubscribers(
      Object instance, Class<T> superEventType) {
    var superTypes = ReflectionUtil.traverseSuperTypes(instance.getClass());
    return Arrays.stream(instance.getClass().getMethods())
        .filter(m -> !Modifier.isStatic(m.getModifiers()))
        .flatMap(method -> Stream.concat(superTypes.stream(), Stream.of(instance.getClass()))
            .flatMap(clazz -> getDeclaredMethod(clazz, method).stream())
            .filter(declaredMethod -> declaredMethod.isAnnotationPresent(Subscribe.class))
            .findFirst()
            .<EventSubscriber<? extends T>>map(
                declaredMethod -> createSubscriber(instance, method, declaredMethod,
                    superEventType))
            .stream());
  }

  @SuppressWarnings("unchecked")
  private static <T> EventSubscriber<? extends T> createSubscriber(
      @Nullable Object instance, Method method, Method declaredMethod, Class<T> superEventType) {
    var parameterTypes = method.getParameterTypes();
    if (parameterTypes.length != 1 && parameterTypes.length != 2) {
      throw new IllegalArgumentException(
          "Method " + method + " has @Subscribe annotation. " +
              "It has " + parameterTypes.length + " argument(s), " +
              "but event handler methods require one or two argument.");
    }

    var eventType = parameterTypes[0];
    if (!superEventType.isAssignableFrom(eventType)
        && !superEventType.isAssignableFrom(eventType)) {
      throw new IllegalArgumentException(
          "Method " + method + " has @Subscribe annotation, " +
              "but takes an argument that is not assignable from: " + superEventType);
    }

    if (parameterTypes.length == 2 && parameterTypes[1] != EventContext.class) {
      throw new IllegalArgumentException(
          "Method " + method + " has @Subscribe annotation, " +
              "but takes a second argument that does not match: " + EventContext.class);
    }

    var annotation = declaredMethod.getAnnotation(Subscribe.class);
    return createSubscriber(instance, declaredMethod, (Class<? extends T>) eventType, annotation);
  }

  private static <T> EventSubscriber<T> createSubscriber(
      @Nullable Object instance, Method method, Class<T> eventType, Subscribe annotation) {
    return new EventSubscriber<T>(eventType,
        EventHandler.<T>builder()
            .ignoreCancelled(annotation.ignoreCancelled())
            .expirationCount(annotation.expirationCount())
            .build(createMethodInvokingHandler(instance, method, eventType)),
        annotation.priority());
  }

  private static Optional<Method> getDeclaredMethod(Class<?> clazz, Method method) {
    try {
      return Optional.of(clazz.getDeclaredMethod(method.getName(), method.getParameterTypes()));
    } catch (NoSuchMethodException e) {
      return Optional.empty();
    }
  }

  private static <T> BiConsumer<T, EventContext> createMethodInvokingHandler(
      @Nullable Object instance, Method listenerMethod, Class<T> eventType) {
    var hasContext = listenerMethod.getParameterTypes().length == 2;
    var handlerAccess = MethodAccess.get(listenerMethod.getDeclaringClass());
    var methodIndex = handlerAccess.getIndex(listenerMethod.getName(),
        hasContext ? new Class[] {eventType, EventContext.class} : new Class[] {eventType});
    return (event, context) -> handlerAccess.invoke(instance, methodIndex,
        hasContext ? new Object[] {event, context} : new Object[] {event});
  }
}
