package flare.eventbus;

import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Predicate;

/**
 * A handler of an event.
 * 
 * @param <T> - event type to handle
 */
@FunctionalInterface
public interface EventHandler<T> {

  /**
   * Handles the specified event.
   * 
   * @param event - event to handle
   * @param context - event context
   * @return the result
   */
  Result handle(T event, EventContext context);

  static <T> Builder<T> builder() {
    return new Builder<>();
  }

  static <T> EventHandler<T> of(Consumer<T> handler) {
    return withResult(handler, Result.SUCCESS);
  }

  static <T> EventHandler<T> withResult(Consumer<T> handler, Result result) {
    return (event, context) -> {
      handler.accept(event);
      return result;
    };
  }

  static <T> EventHandler<T> of(BiConsumer<T, EventContext> handler) {
    return withResult(handler, Result.SUCCESS);
  }

  static <T> EventHandler<T> withResult(BiConsumer<T, EventContext> handler, Result result) {
    return (event, context) -> {
      handler.accept(event, context);
      return result;
    };
  }

  class Builder<T> {

    private final List<Predicate<T>> filters = new ArrayList<>();
    private boolean ignoreCancelled = true;
    private int expirationCount;
    private Predicate<T> expireWhen;

    protected Builder() {}

    /**
     * Adds a filter to the executor of this listener. The executor will only be called if this
     * condition passes on the given event.
     */
    public Builder<T> filter(Predicate<T> filter) {
      this.filters.add(filter);
      return this;
    }

    /**
     * Specifies if the handler should still be called if {@link EventContext#isCancelled()} returns
     * {@code true}.
     * <p>
     * Default is set to {@code true}.
     *
     * @param ignoreCancelled True to stop processing the event when cancelled
     */
    public EventHandler.Builder<T> ignoreCancelled(boolean ignoreCancelled) {
      this.ignoreCancelled = ignoreCancelled;
      return this;
    }

    /**
     * Removes this listener after it has been executed the given number of times.
     *
     * @param expirationCount The number of times to execute
     */
    public EventHandler.Builder<T> expirationCount(int expirationCount) {
      this.expirationCount = expirationCount;
      return this;
    }

    /**
     * Expires this listener when it passes the given condition. The expiration will happen before
     * the event is executed.
     *
     * @param expireWhen The condition to test
     */
    public EventHandler.Builder<T> expireWhen(Predicate<T> expireWhen) {
      this.expireWhen = expireWhen;
      return this;
    }

    @SuppressWarnings("unchecked")
    public EventHandler<T> build(Consumer<T> handler) {
      return new BuiltEventListener<>(
          this.ignoreCancelled,
          this.expirationCount > 0 ? new AtomicInteger(this.expirationCount) : null,
          this.expireWhen,
          this.filters.isEmpty() ? null : this.filters.toArray(Predicate[]::new),
          handler);
    }
  }

  enum Result {

    SUCCESS,
    INVALID,
    EXPIRED;
  }
}
