package flare.eventbus;

import java.util.Collections;
import java.util.IdentityHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import org.jetbrains.annotations.Nullable;
import it.unimi.dsi.fastutil.objects.ReferenceOpenHashSet;

final class EventBusImpl<T> implements EventBus<T> {

  private final Class<T> eventType;
  @Nullable
  private final ClassHierarchyService hierarchyService;

  private final Map<Object, EventSubscriber<? extends T>[]> annotatedSubscribers =
      new IdentityHashMap<>();

  private final Map<Class<? extends T>, EventDispatcher<? extends T>> dispatchers =
      new IdentityHashMap<>();

  private final Set<EventBusImpl<? extends T>> children = new ReferenceOpenHashSet<>();
  @SuppressWarnings("unchecked")
  private EventBus<? extends T>[] activeChildren = new EventBus[0];

  @Nullable
  private EventBusImpl<? super T> parent;
  private boolean active;

  EventBusImpl(Class<T> eventType, @Nullable ClassHierarchyService hierarchyService) {
    this.eventType = eventType;
    this.hierarchyService = hierarchyService;
  }

  private <E extends T> boolean dispatchAs(
      Class<? super E> eventType, E event, EventContext context) {
    @SuppressWarnings("unchecked")
    var dispatcher = (EventDispatcher<? super E>) this.dispatchers.get(eventType);
    if (dispatcher != null) {
      dispatcher.dispatch(event, context);
      return !dispatcher.hasSubscribers();
    }
    return false;
  }

  @SuppressWarnings("unchecked")
  @Override
  public <E extends T> EventContext publish(E event, EventContext context) {
    var eventType = typeOf(event);

    for (int i = 0; i < this.activeChildren.length; i++) {
      var child = this.activeChildren[i];
      if (child.eventType().isAssignableFrom(eventType)) {
        ((EventBusImpl<? super E>) child).publish(event, context);
        if (context.isCancelled()) {
          return context;
        }
      }
    }

    var checkActive = false;
    checkActive |= this.dispatchAs(eventType, event, context);
    if (context.isCancelled()) {
      return context;
    }

    if (this.hierarchyService != null) {
      var superTypes = this.hierarchyService.superTypesOf(eventType);
      for (int i = 0; i < superTypes.length; i++) {
        var superType = superTypes[i];
        checkActive |= this.dispatchAs(superTypes[i], event, context);
        if (context.isCancelled()) {
          return context;
        }
        if (superType == this.eventType) {
          break;
        }
      }
    }

    if (checkActive) {
      this.checkActive();
    }

    return context;
  }

  @Override
  public Class<T> eventType() {
    return this.eventType;
  }

  @Override
  public Optional<EventBus<? super T>> parent() {
    return Optional.ofNullable(this.parent);
  }

  @Override
  public Set<EventBus<? extends T>> children() {
    return Collections.unmodifiableSet(this.children);
  }

  @Override
  public <E extends T> EventBus<T> addChild(EventBus<E> child) {
    var childImpl = asImpl(child);
    if (this.children.add(childImpl)) {
      childImpl.parent = this;
      this.updateActiveChildren();
      this.checkActive();
    }
    return this;
  }

  @Override
  public <E extends T> EventBus<T> removeChild(EventBus<E> child) {
    var childImpl = asImpl(child);
    if (this.children.remove(childImpl)) {
      childImpl.parent = null;
      this.updateActiveChildren();
      this.checkActive();
    }
    return this;
  }

  @SuppressWarnings("unchecked")
  @Override
  public EventBus<T> subscribeAnnotated(Object obj) {
    this.annotatedSubscribers.computeIfAbsent(obj,
        __ -> AnnotatedSubscriberUtil.findAnnotatedSubscribers(obj, this.eventType)
            .peek(this::subscribe)
            .toArray(EventSubscriber[]::new));
    return this;
  }

  @Override
  public EventBus<T> unsubscribeAnnotated(Object obj) {
    var listeners = this.annotatedSubscribers.remove(obj);
    if (listeners != null) {
      for (int i = 0; i < listeners.length; i++) {
        this.unsubscribe(listeners[i]);
      }
    }
    return this;
  }

  @Override
  public <E extends T> EventBus<T> subscribe(EventSubscriber<E> subscriber) {
    @SuppressWarnings("unchecked")
    var dispatcher = (EventDispatcher<E>) this.dispatchers.get(subscriber.eventType());
    if (dispatcher == null) {
      dispatcher = new HashEventDispatcher<>();
      this.dispatchers.put(subscriber.eventType(), dispatcher);
      this.checkActive();
    }
    dispatcher.register(subscriber);
    return this;
  }

  @Override
  public <E extends T> EventBus<T> unsubscribe(EventSubscriber<E> subscriber) {
    @SuppressWarnings("unchecked")
    var dispatcher = (EventDispatcher<E>) this.dispatchers.get(subscriber.eventType());
    if (dispatcher == null) {
      return this;
    }
    if (dispatcher.unregister(subscriber) && !dispatcher.hasSubscribers()) {
      this.dispatchers.remove(subscriber.eventType());
      this.checkActive();
    }
    return this;
  }

  private void checkActive() {
    var active = this.activeChildren.length > 0 || !this.dispatchers.isEmpty();
    if (active == this.active) {
      return;
    }
    this.active = active;
    if (this.parent != null) {
      this.parent.updateActiveChildren();
    }
  }

  private boolean isActive() {
    return this.active;
  }

  @SuppressWarnings("unchecked")
  private void updateActiveChildren() {
    this.activeChildren = this.children.stream()
        .filter(EventBusImpl::isActive)
        .toArray(EventBusImpl[]::new);
  }

  @SuppressWarnings("unchecked")
  private static <T> Class<T> typeOf(T object) {
    return (Class<T>) object.getClass();
  }

  private static <T> EventBusImpl<T> asImpl(EventBus<T> eventNode) {
    return (EventBusImpl<T>) eventNode;
  }
}
