package flare.eventbus;

import java.lang.reflect.Method;
import java.lang.reflect.Modifier;
import java.util.Arrays;
import java.util.Optional;
import java.util.function.Consumer;
import java.util.stream.Stream;
import org.jetbrains.annotations.Nullable;
import com.esotericsoftware.reflectasm.MethodAccess;
import flare.commons.ReflectionUtil;

final class AnnotatedSubscriberUtil {

  private AnnotatedSubscriberUtil() {}

  static <T> Stream<EventSubscriber<? extends T>> findAnnotatedSubscribers(
      Object obj, Class<T> superEventType) {
    return obj.getClass() == Class.class
        ? findStaticAnnotatedSubscribers((Class<?>) obj, superEventType)
        : findInstanceAnnotatedSubscribers(obj, superEventType);
  }

  private static <T> Stream<EventSubscriber<? extends T>> findStaticAnnotatedSubscribers(
      Class<?> clazz, Class<T> superEventType) {
    return Arrays.stream(clazz.getMethods())
        .filter(method -> Modifier.isStatic(method.getModifiers()))
        .filter(method -> method.isAnnotationPresent(Subscribe.class))
        .map(method -> createSubscriber(null, method, method, superEventType));
  }

  private static <T> Stream<EventSubscriber<? extends T>> findInstanceAnnotatedSubscribers(
      Object instance, Class<T> superEventType) {
    var superTypes = ReflectionUtil.traverseSuperTypes(instance.getClass());
    return Arrays.stream(instance.getClass().getMethods())
        .filter(m -> !Modifier.isStatic(m.getModifiers()))
        .flatMap(method -> Stream.concat(superTypes.stream(), Stream.of(instance.getClass()))
            .flatMap(clazz -> getDeclaredMethod(clazz, method).stream())
            .filter(declaredMethod -> declaredMethod.isAnnotationPresent(Subscribe.class))
            .findFirst()
            .<EventSubscriber<? extends T>>map(
                declaredMethod -> createSubscriber(instance, method, declaredMethod,
                    superEventType))
            .stream());
  }

  @SuppressWarnings("unchecked")
  private static <T> EventSubscriber<? extends T> createSubscriber(
      @Nullable Object instance, Method method, Method declaredMethod, Class<T> superEventType) {
    var parameterTypes = method.getParameterTypes();
    if (parameterTypes.length != 1) {
      throw new IllegalArgumentException(
          "Method " + method + " has @Subscribe annotation. " +
              "It has " + parameterTypes.length + " arguments, " +
              "but event handler methods require a single argument only.");
    }

    var eventType = parameterTypes[0];
    if (!superEventType.isAssignableFrom(eventType)
        && !superEventType.isAssignableFrom(eventType)) {
      throw new IllegalArgumentException(
          "Method " + method + " has @Subscribe annotation, " +
              "but takes an argument that is not assignable from: " + superEventType);
    }

    var annotation = declaredMethod.getAnnotation(Subscribe.class);
    return createSubscriber(instance, declaredMethod, (Class<? extends T>) eventType, annotation);
  }

  private static <T> EventSubscriber<T> createSubscriber(
      @Nullable Object instance, Method method, Class<T> eventType, Subscribe annotation) {
    return new EventSubscriber<T>(eventType,
        EventHandler.<T>builder()
            .ignoreCancelled(annotation.ignoreCancelled())
            .expirationCount(annotation.expirationCount())
            .build(createMethodInvokingHandler(instance, method, eventType)),
        annotation.priority());
  }

  private static Optional<Method> getDeclaredMethod(Class<?> clazz, Method method) {
    try {
      return Optional.of(clazz.getDeclaredMethod(method.getName(), method.getParameterTypes()));
    } catch (NoSuchMethodException e) {
      return Optional.empty();
    }
  }

  private static <T> Consumer<T> createMethodInvokingHandler(
      @Nullable Object instance, Method listenerMethod, Class<T> eventType) {
    var handlerAccess = MethodAccess.get(listenerMethod.getDeclaringClass());
    var methodIndex = handlerAccess.getIndex(listenerMethod.getName(), eventType);
    return event -> handlerAccess.invoke(instance, methodIndex, event);
  }
}
