package flare.eventbus;

import java.util.Optional;
import java.util.Set;
import java.util.function.Consumer;
import org.jetbrains.annotations.Nullable;

/**
 * An event bus which dispatches events to subscribed handlers.
 * 
 * @param <T> - root event type
 */
public sealed interface EventBus<T> permits EventBusImpl {

  static EventBus<Object> create() {
    return create(Object.class);
  }

  static <T> EventBus<T> create(Class<T> eventType) {
    return create(eventType, true);
  }

  static <T> EventBus<T> create(Class<T> eventType, boolean hierarchicalDispatch) {
    return create(eventType,
        hierarchicalDispatch ? ClassHierarchyService.cached() : null);
  }

  static <T> EventBus<T> create(Class<T> eventType,
      @Nullable ClassHierarchyService hierarchyService) {
    return new EventBusImpl<>(eventType, hierarchyService);
  }

  /**
   * Post an event starting from this node.
   *
   * @param event the event to call
   */
  <E extends T> boolean post(E event);

  Class<T> eventType();

  Optional<EventBus<? super T>> parent();

  Set<EventBus<? extends T>> children();

  <E extends T> EventBus<T> addChild(EventBus<E> child);

  <E extends T> EventBus<T> removeChild(EventBus<E> child);

  /**
   * Subscribes all method annotated with {@link Subscribe} declared in the specified object.
   * Passing a {@link Class} to this method will subscribe static methods, passing an instance will
   * subscribe instance methods.
   * 
   * @param object - an object instance or a class
   * @return the event bus
   */
  EventBus<T> subscribeAnnotated(Object object);

  /**
   * Unsubscribes all methods annotated with {@link Subscribe} declared in the specified object.
   * Passing a {@link Class} to this method will unsubscribe static methods, passing an instance
   * will unsubscribe instance methods.
   * 
   * @param object - an object instance or a class
   * @return the event bus
   */
  EventBus<T> unsubscribeAnnotated(Object object);

  default <E extends T> EventBus<T> subscribe(Class<E> eventType, Consumer<E> handler) {
    return this.subscribe(EventSubscriber.of(eventType, handler));
  }

  <E extends T> EventBus<T> subscribe(EventSubscriber<E> subscriber);

  <E extends T> EventBus<T> unsubscribe(EventSubscriber<E> subscriber);
}
