package flare.commons;

import java.io.IOException;
import java.io.UncheckedIOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.EnumSet;
import java.util.List;
import java.util.Optional;
import java.util.Queue;
import java.util.Set;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.function.BooleanSupplier;
import java.util.function.Consumer;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.function.Supplier;
import java.util.function.ToIntFunction;
import java.util.random.RandomGenerator;

public final class FunctionalUtil {

  private FunctionalUtil() {}

  private static final Consumer<Object> EMPTY_CONSUMER = __ -> {};

  @FunctionalInterface
  public interface ExceptionalRunnable<E extends Exception> {

    void run() throws E;
  }

  public static Runnable uncheckException(ExceptionalRunnable<? extends IOException> runnable) {
    return uncheckException(IOException.class, runnable, UncheckedIOException::new);
  }

  public static <E extends Exception> Runnable uncheckException(
      Class<E> exceptionType, ExceptionalRunnable<? extends E> runnable,
      Function<E, ? extends RuntimeException> exceptionMapper) {
    return () -> {
      try {
        runnable.run();
      } catch (Exception e) {
        throw exceptionType.isInstance(e)
            ? exceptionMapper.apply(exceptionType.cast(e))
            : new IllegalStateException(e);
      }
    };
  }

  public static <T, P extends Enum<P>> Predicate<T> distinctEnum(
      Function<T, P> propertyExtractor, Consumer<T> dropHandler, Class<P> enumType) {
    return distinct(propertyExtractor, dropHandler, () -> EnumSet.noneOf(enumType));
  }

  public static <T, P> Predicate<T> distinct(Function<T, P> propertyExtractor,
      Consumer<T> dropHandler, Supplier<? extends Set<P>> setFactory) {
    Set<P> seen = setFactory.get();
    return t -> {
      var result = seen.add(propertyExtractor.apply(t));
      if (!result) {
        dropHandler.accept(t);
      }
      return result;
    };
  }

  public static <T> T findNext(Queue<CompletableFuture<T>> queue, BooleanSupplier hasWork) {
    var shouldStealWork = true;

    while (!queue.isEmpty()) {
      var iterator = queue.iterator();

      while (iterator.hasNext()) {
        var future = iterator.next();

        if (shouldStealWork && !future.isDone()) {
          continue;
        }

        iterator.remove();

        try {
          return future.join();
        } catch (CancellationException e) {}
      }

      if (shouldStealWork && !hasWork.getAsBoolean()) {
        shouldStealWork = false;
      }
    }

    return null;
  }

  @SuppressWarnings("unchecked")
  public static <T> Consumer<T> noopConsumer() {
    return (Consumer<T>) EMPTY_CONSUMER;
  }

  public static <T extends Enum<T>> int mapToBits(Collection<T> values,
      ToIntFunction<T> bitMapper) {
    return values.stream()
        .mapToInt(bitMapper)
        .reduce(0, (a, b) -> a | b);
  }

  public static <V> CompletableFuture<List<V>> sequenceFailFast(
      List<? extends CompletableFuture<? extends V>> futures) {
    List<V> results = new ArrayList<>(futures.size());
    var newFutures = new CompletableFuture[futures.size()];
    CompletableFuture<Void> exceptionalFuture = new CompletableFuture<>();
    futures.forEach(future -> {
      int futureIndex = results.size();
      results.add(null);
      newFutures[futureIndex] = future.whenComplete((result, exception) -> {
        if (exception != null) {
          exceptionalFuture.completeExceptionally(exception);
        } else {
          results.set(futureIndex, result);
        }
      });
    });
    return CompletableFuture.allOf(newFutures)
        .applyToEither(exceptionalFuture, __ -> results);
  }

  public static <T> T make(Supplier<T> supplier) {
    return supplier.get();
  }

  public static <T> T make(T instance, Consumer<T> consumer) {
    consumer.accept(instance);
    return instance;
  }

  public static float[] filledArray(int length, float value) {
    var array = new float[length];
    Arrays.fill(array, value);
    return array;
  }

  public static int[] filledArray(int length, int value) {
    var array = new int[length];
    Arrays.fill(array, value);
    return array;
  }

  public static <T> T getRandom(List<T> list, RandomGenerator random) {
    return list.get(random.nextInt(list.size()));
  }

  public static <T> Optional<T> getRandomSafe(List<T> list, RandomGenerator random) {
    return list.isEmpty() ? Optional.empty() : Optional.of(getRandom(list, random));
  }

  public static <T> T getRandom(T[] array, RandomGenerator random) {
    return array[random.nextInt(array.length)];
  }
}
