/*
 * Decompiled with CFR 0.152.
 */
package com.craftingdead.core.telemetry;

import com.craftingdead.core.CommonConfig;
import com.craftingdead.core.CraftingDead;
import com.craftingdead.core.telemetry.TelemetryEnvironment;
import com.craftingdead.core.telemetry.TelemetryRuntimeSampler;
import com.google.gson.Gson;
import com.google.gson.GsonBuilder;
import com.mojang.logging.LogUtils;
import io.sentry.Scope;
import io.sentry.Sentry;
import io.sentry.SentryOptions;
import io.sentry.protocol.User;
import java.net.InetAddress;
import java.net.UnknownHostException;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.time.Duration;
import java.util.ArrayList;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Supplier;
import net.minecraft.SharedConstants;
import net.minecraftforge.api.distmarker.Dist;
import net.minecraftforge.fml.ModList;
import net.minecraftforge.fml.loading.FMLEnvironment;
import net.minecraftforge.versions.forge.ForgeVersion;
import org.apache.commons.lang3.StringUtils;
import org.slf4j.Logger;

public final class TelemetryManager {
    private static final Logger LOGGER = LogUtils.getLogger();
    private static final AtomicBoolean INITIALIZATION_ATTEMPTED = new AtomicBoolean();
    private static final Set<String> REGISTERED_MODULES = ConcurrentHashMap.newKeySet();
    private static final AtomicBoolean SAMPLER_STARTED = new AtomicBoolean();
    private static final Gson GSON = new GsonBuilder().disableHtmlEscaping().create();

    private TelemetryManager() {
    }

    public static boolean initialize(String moduleId, String moduleVersion, Supplier<Optional<String>> fallbackDsnSupplier, Consumer<SentryOptions> optionsCustomizer, Consumer<Scope> scopeCustomizer) {
        Objects.requireNonNull(moduleId, "moduleId");
        Objects.requireNonNull(moduleVersion, "moduleVersion");
        if (!Sentry.isEnabled() && INITIALIZATION_ATTEMPTED.compareAndSet(false, true)) {
            TelemetryManager.attemptInitialization(fallbackDsnSupplier, optionsCustomizer);
        }
        if (!Sentry.isEnabled()) {
            return false;
        }
        TelemetryManager.registerModule(moduleId, moduleVersion, scopeCustomizer);
        return true;
    }

    public static boolean initialize(String moduleId, String moduleVersion, Supplier<Optional<String>> fallbackDsnSupplier, Consumer<Scope> scopeCustomizer) {
        return TelemetryManager.initialize(moduleId, moduleVersion, fallbackDsnSupplier, options -> {}, scopeCustomizer);
    }

    private static void attemptInitialization(Supplier<Optional<String>> fallbackDsnSupplier, Consumer<SentryOptions> optionsCustomizer) {
        String dsn;
        CommonConfig config = CommonConfig.instance;
        LOGGER.info("Telemetry initialization requested for {} runtime", (Object)FMLEnvironment.dist);
        if (!TelemetryManager.hasRuntimeConsent(config)) {
            LOGGER.info("Telemetry disabled for {} runtime", (Object)FMLEnvironment.dist);
            return;
        }
        boolean passwordValid = TelemetryManager.validatePassword(config);
        if (!passwordValid) {
            LOGGER.debug("Telemetry password validation failed or bypassed; continuing with initialization");
        }
        if ((dsn = TelemetryManager.resolveDsn(config, fallbackDsnSupplier)).isBlank()) {
            LOGGER.warn("Telemetry blocked because no DSN was configured");
            return;
        }
        TelemetryEnvironment environment = (TelemetryEnvironment)((Object)config.telemetryEnvironment.get());
        double tracesSampleRate = (Double)config.telemetryTracesSampleRate.get();
        String release = TelemetryManager.resolveBuildId();
        LOGGER.info("Telemetry initializing Sentry with environment {} release {} and trace sample rate {}", new Object[]{environment, release, tracesSampleRate});
        try {
            Sentry.init(options -> {
                options.setDsn(dsn);
                options.setEnvironment(environment.getWireValue());
                options.setRelease(release);
                options.setTracesSampleRate(Double.valueOf(tracesSampleRate));
                options.setDebug(false);
                options.setAttachStacktrace(true);
                options.setSendDefaultPii(true);
                options.setMaxBreadcrumbs(200);
                options.setEnableAutoSessionTracking(true);
                options.setSessionTrackingIntervalMillis((long)((int)Duration.ofMinutes(1L).toMillis()));
                options.setMaxQueueSize(100);
                options.setServerName(TelemetryManager.resolveHostName());
                if (optionsCustomizer != null) {
                    optionsCustomizer.accept(options);
                }
            });
        }
        catch (RuntimeException e) {
            LOGGER.error("Failed to initialize telemetry", (Throwable)e);
            return;
        }
        if (!Sentry.isEnabled()) {
            LOGGER.warn("Telemetry initialization completed but Sentry is not enabled");
            return;
        }
        Sentry.configureScope(scope -> TelemetryManager.decorateBaseScope(scope, environment, release));
        LOGGER.info("Telemetry enabled for environment {} with release {}", (Object)environment, (Object)release);
        TelemetryManager.ensureSamplerStarted();
    }

    private static void registerModule(String moduleId, String moduleVersion, Consumer<Scope> scopeCustomizer) {
        Sentry.configureScope(scope -> {
            if (REGISTERED_MODULES.add(moduleId)) {
                scope.setTag(moduleId + ".version", moduleVersion);
                scope.setExtra(moduleId + ".version", moduleVersion);
            }
            scope.setTag("modules", String.join((CharSequence)",", REGISTERED_MODULES));
            if (scopeCustomizer != null) {
                scopeCustomizer.accept(scope);
            }
        });
    }

    public static void updateUser(User user, Map<String, String> tagValues, Map<String, String> extraValues) {
        if (user == null || !Sentry.isEnabled()) {
            return;
        }
        Sentry.configureScope(scope -> {
            scope.setUser(user);
            if (StringUtils.isNotBlank((CharSequence)user.getId())) {
                scope.setTag("user.id", user.getId());
            }
            if (StringUtils.isNotBlank((CharSequence)user.getUsername())) {
                scope.setTag("user.username", user.getUsername());
            }
            if (StringUtils.isNotBlank((CharSequence)user.getEmail())) {
                scope.setTag("user.email", user.getEmail());
            }
            if (tagValues != null && !tagValues.isEmpty()) {
                tagValues.forEach((key, value) -> {
                    if (StringUtils.isNotBlank((CharSequence)key) && StringUtils.isNotBlank((CharSequence)value)) {
                        scope.setTag(key, value);
                    }
                });
            }
            if (extraValues != null && !extraValues.isEmpty()) {
                extraValues.forEach((key, value) -> {
                    if (StringUtils.isNotBlank((CharSequence)key) && value != null) {
                        scope.setExtra(key, value);
                    }
                });
            }
        });
    }

    private static boolean hasRuntimeConsent(CommonConfig config) {
        Dist dist = FMLEnvironment.dist;
        if (dist == Dist.DEDICATED_SERVER) {
            return (Boolean)config.telemetryServerEnabled.get();
        }
        return (Boolean)config.telemetryClientEnabled.get();
    }

    private static boolean validatePassword(CommonConfig config) {
        String expectedHash = StringUtils.defaultString((String)((String)config.telemetryPasswordHash.get()));
        if (expectedHash.isBlank()) {
            LOGGER.debug("Telemetry password hash not configured; telemetry will initialize without password gate");
            return true;
        }
        Optional<String> passwordSecret = TelemetryManager.resolveSecret();
        if (passwordSecret.isEmpty()) {
            LOGGER.debug("Telemetry password secret not provided. Supply craftingdead.sentryPassword system property or CD_SENTRY_PASSWORD environment variable");
            return false;
        }
        String salt = StringUtils.defaultString((String)((String)config.telemetryPasswordSalt.get()));
        String computedHash = TelemetryManager.hashWithSalt(salt, passwordSecret.get());
        if (!expectedHash.equalsIgnoreCase(computedHash)) {
            LOGGER.warn("Telemetry password secret does not match configured hash");
            return false;
        }
        return true;
    }

    private static Optional<String> resolveSecret() {
        String propertySecret = System.getProperty("craftingdead.sentryPassword");
        if (StringUtils.isNotBlank((CharSequence)propertySecret)) {
            return Optional.of(propertySecret);
        }
        String envSecret = System.getenv("CD_SENTRY_PASSWORD");
        if (StringUtils.isNotBlank((CharSequence)envSecret)) {
            return Optional.of(envSecret);
        }
        String envSecretAlt = System.getenv("CRAFTING_DEAD_SENTRY_PASSWORD");
        if (StringUtils.isNotBlank((CharSequence)envSecretAlt)) {
            return Optional.of(envSecretAlt);
        }
        return Optional.empty();
    }

    private static String hashWithSalt(String salt, String secret) {
        String payload = salt.isEmpty() ? secret : salt + ":" + secret;
        try {
            MessageDigest digest = MessageDigest.getInstance("SHA-256");
            byte[] hash = digest.digest(payload.getBytes(StandardCharsets.UTF_8));
            return TelemetryManager.toHex(hash);
        }
        catch (NoSuchAlgorithmException e) {
            throw new IllegalStateException("SHA-256 digest not available", e);
        }
    }

    private static String toHex(byte[] data) {
        StringBuilder builder = new StringBuilder(data.length * 2);
        for (byte value : data) {
            builder.append(String.format(Locale.ROOT, "%02x", value));
        }
        return builder.toString();
    }

    private static String resolveDsn(CommonConfig config, Supplier<Optional<String>> fallbackDsnSupplier) {
        String configured;
        TelemetryEnvironment environment = (TelemetryEnvironment)((Object)config.telemetryEnvironment.get());
        String string = configured = environment == TelemetryEnvironment.PRODUCTION ? (String)config.telemetryDsnProduction.get() : (String)config.telemetryDsnExperimental.get();
        if (StringUtils.isBlank((CharSequence)configured)) {
            configured = System.getProperty("craftingdead.sentryDsn", "");
        }
        if (StringUtils.isBlank((CharSequence)configured)) {
            configured = System.getenv("CRAFTING_DEAD_SENTRY_DSN");
        }
        if (StringUtils.isBlank((CharSequence)configured)) {
            configured = System.getenv("SENTRY_DSN");
        }
        if (StringUtils.isBlank((CharSequence)configured) && fallbackDsnSupplier != null) {
            configured = fallbackDsnSupplier.get().orElse("");
        }
        return StringUtils.defaultString((String)configured);
    }

    private static void decorateBaseScope(Scope scope, TelemetryEnvironment environment, String release) {
        String modpackId;
        scope.setTag("environment", environment.getWireValue());
        scope.setTag("runtime", TelemetryManager.resolveRuntimeTag());
        scope.setTag("dist", FMLEnvironment.dist.name());
        scope.setTag("forgeVersion", ForgeVersion.getVersion());
        scope.setTag("mcVersion", SharedConstants.m_183709_().getName());
        scope.setTag("buildId", release);
        scope.setTag("os.arch", System.getProperty("os.arch"));
        scope.setTag("os.version", System.getProperty("os.version"));
        scope.setTag("user.language", System.getProperty("user.language"));
        scope.setTag("jvm.version", System.getProperty("java.runtime.version"));
        scope.setTag("jvm.vendor", System.getProperty("java.vendor"));
        String commit = TelemetryManager.resolveCommitId();
        if (StringUtils.isNotBlank((CharSequence)commit)) {
            scope.setTag("commit", commit);
        }
        if (StringUtils.isNotBlank((CharSequence)(modpackId = StringUtils.defaultString((String)((String)CommonConfig.instance.telemetryModpackId.get()))))) {
            scope.setTag("modpackId", modpackId);
        }
        Runtime runtime = Runtime.getRuntime();
        scope.setExtra("runtime.memory", TelemetryManager.encodeToJson(TelemetryManager.mapOf("freeBytes", runtime.freeMemory(), "totalBytes", runtime.totalMemory(), "maxBytes", runtime.maxMemory())));
        scope.setExtra("runtime.threads", Integer.toString(Thread.activeCount()));
        scope.setExtra("system.properties", TelemetryManager.encodeToJson(TelemetryManager.collectSystemProperties()));
        scope.setExtra("system.user", TelemetryManager.encodeToJson(TelemetryManager.mapOf("name", System.getProperty("user.name"), "home", System.getProperty("user.home"), "dir", System.getProperty("user.dir"))));
        scope.setExtra("mods.loaded", TelemetryManager.encodeToJson(TelemetryManager.collectModSnapshot()));
    }

    private static String resolveRuntimeTag() {
        return FMLEnvironment.dist == Dist.DEDICATED_SERVER ? "server" : "client";
    }

    private static String resolveBuildId() {
        String buildId = TelemetryManager.firstNonBlank(System.getenv("CRAFTING_DEAD_BUILD_ID"), System.getProperty("craftingdead.buildId"), CraftingDead.VERSION);
        return (String)StringUtils.defaultIfBlank((CharSequence)buildId, (CharSequence)CraftingDead.VERSION);
    }

    private static String resolveCommitId() {
        return TelemetryManager.firstNonBlank(System.getenv("GITHUB_SHA"), System.getenv("CRAFTING_DEAD_COMMIT"), System.getProperty("craftingdead.commit"));
    }

    private static String firstNonBlank(String ... values) {
        for (String value : values) {
            if (!StringUtils.isNotBlank((CharSequence)value)) continue;
            return value;
        }
        return "";
    }

    private static void ensureSamplerStarted() {
        if (SAMPLER_STARTED.compareAndSet(false, true)) {
            TelemetryRuntimeSampler.ensureStarted();
        }
    }

    private static String resolveHostName() {
        try {
            return InetAddress.getLocalHost().getHostName();
        }
        catch (UnknownHostException e) {
            LOGGER.debug("Unable to resolve hostname", (Throwable)e);
            return "unknown-host";
        }
    }

    private static Map<String, String> collectSystemProperties() {
        LinkedHashMap<String, String> properties = new LinkedHashMap<String, String>();
        System.getProperties().forEach((BiConsumer<? super Object, ? super Object>)((BiConsumer<Object, Object>)(key, value) -> {
            if (key != null && value != null) {
                properties.put(String.valueOf(key), String.valueOf(value));
            }
        }));
        return properties;
    }

    private static List<Map<String, String>> collectModSnapshot() {
        ArrayList<Map<String, String>> mods = new ArrayList<Map<String, String>>();
        ModList.get().getMods().forEach(mod -> {
            LinkedHashMap<String, String> entry = new LinkedHashMap<String, String>();
            entry.put("id", mod.getModId());
            entry.put("displayName", mod.getDisplayName());
            entry.put("version", mod.getVersion().toString());
            mods.add(entry);
        });
        return mods;
    }

    static String encodeToJson(Object value) {
        try {
            return GSON.toJson(value);
        }
        catch (RuntimeException e) {
            LOGGER.warn("Failed to encode telemetry payload to JSON", (Throwable)e);
            return String.valueOf(value);
        }
    }

    private static Map<String, Object> mapOf(Object ... entries) {
        LinkedHashMap<String, Object> map = new LinkedHashMap<String, Object>();
        int i = 0;
        while (i + 1 < entries.length) {
            map.put(String.valueOf(entries[i]), entries[i + 1]);
            i += 2;
        }
        return map;
    }
}

